#include <ifractal.h>
#include <ifdevice.h>
#include <tunnel.h>

#include <curl/curl.h>

void tunnel_session_free_down(TUNNEL_SESSION *);
void tunnel_session_free_up(TUNNEL_SESSION *);

extern IF_GETOPT configs[];

IF_THREAD_LIST *streams;
IFLOG logger;


// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_server(TUNNEL_SESSION *sess, char *value)
{
	if (strncmp(value, TUNNEL_AGENT, sizeof(TUNNEL_AGENT) - 1) == 0)
	{
		sess->type = TUNNEL_SESSION_WS;
	}
	else
	{
		sess->type = TUNNEL_SESSION_ERROR;
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_remote(TUNNEL_SESSION *sess, char *value)
{
	strncpy(sess->header_cmd, value, PATH_LEN);
	sess->type = TUNNEL_SESSION_STREAM;
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_stream(TUNNEL_SESSION *sess, char *value)
{
	strncpy(sess->header_cmd, value, PATH_LEN);
	sess->type = TUNNEL_SESSION_STREAM;
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_timeout(TUNNEL_SESSION *sess, char *value)
{
	sess->timeout = atoi(value);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_token(TUNNEL_SESSION *sess, char *value)
{
	strncpy(sess->token, value, PATH_LEN);
	for (int i = 0 ; sess->token[i] != 0 ; i++)
		if ((sess->token[i] == '\r') || (sess->token[i] == '\n'))
			sess->token[i] = 0;
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_header_handler_length(TUNNEL_SESSION *sess, char *value)
{
	sess->down_pos = 0;
	sess->down_len = atoi(value);
	sess->down_data = if_malloc(sess->down_len + 1);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
_CALLBACK size_t curl_header_cb(char *buf, size_t size, size_t nmemb, void *userdata)
{
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) userdata;
	size_t len = size * nmemb;
	int prefix_len;
	char *prefix;
	struct
	{
		char *header;
		int (*handler)(TUNNEL_SESSION *, char *);
	} headers[] = {
		{HTTP_HEADER_LENGTH, tunnel_header_handler_length},
		{TOKEN_HEADER, tunnel_header_handler_token},
		{TIMEOUT_HEADER, tunnel_header_handler_timeout},
		{REMOTE_HEADER, tunnel_header_handler_remote},
		{STREAM_HEADER, tunnel_header_handler_stream},
		{SERVER_HEADER, tunnel_header_handler_server},
		{NULL, NULL}
	};

	for (int i = 0 ; headers[i].header != NULL ; i++)
	{
		prefix = headers[i].header;
		prefix_len = strlen(prefix);

		if (strncmp(prefix, buf, prefix_len) == 0)
			headers[i].handler(sess, buf + prefix_len);
	}

	if (logger.verbosity >= IF_VERB_DEBUG)
		fwrite(buf, size, nmemb, stdout);

	return(len);
}
// ///////////////////////////////////////////////////////////////////// //
_CALLBACK size_t curl_write_cb(void *ptr, size_t size, size_t nmemb, void *data)
{
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) data;
	uint8_t *buf = (uint8_t *) ptr;
	size_t len = size * nmemb;

	if (logger.verbosity >= IF_VERB_DEBUG)
		fwrite(buf, size, nmemb, stdout);

	if (sess->down_len == 0)
		return(len);

	if (len > (sess->down_len - sess->down_pos))
		len = sess->down_len - sess->down_pos; 

	memcpy(sess->down_data + sess->down_pos, buf, len);
	sess->down_pos += len;
	sess->down_data[sess->down_pos] = 0;

	return(len);
}
// ///////////////////////////////////////////////////////////////////// //
_CALLBACK size_t curl_read_cb(void *ptr, size_t size, size_t nmemb, void *data)
{
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) data;
	uint8_t *buf = (uint8_t *) ptr;
	size_t len = size * nmemb;

	if (len > (sess->up_len - sess->up_pos))
		len = sess->up_len - sess->up_pos; 

	memcpy(buf, sess->up_data + sess->up_pos, len);
	sess->up_pos += len;

	return(len);
}
// ///////////////////////////////////////////////////////////////////// //
_CALLBACK int curl_progress_cb(void *user_data, double dltotal, double dlnow, double ultotal, double ulnow)
{
	//TUNNEL_SESSION *sess = (TUNNEL_SESSION *) user_data;
	
	//verboseDEBUG(&logger, "Session: %lX   Down: %lg/%lg  -  Up: %lg/%lg\n", 
	//	(uint64_t) sess, dlnow, dltotal, ulnow, ultotal);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_post(TUNNEL_SESSION *sess) 
{
	struct curl_slist *chunk = NULL;
	time_t now = time(NULL);
	char buf[BUFFER_LEN];
	CURL *curl;
	CURLcode res;

	curl = curl_easy_init();
	if (curl == NULL)
	{
		verboseFATAL(&logger, "Falha ao tentar iniciar libCurl.\n");
		return(-1);
	}

	snprintf(buf, PATH_LEN, "Unixtime: %ld", now);
	chunk = curl_slist_append(chunk, buf);

	curl_easy_setopt(curl, CURLOPT_HTTPHEADER, chunk);

	curl_easy_setopt(curl, CURLOPT_POST, 1L);
	curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, sess->up_len);

	// Ignore PEER verification
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);

	//curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
	//curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);

	curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, curl_progress_cb);
	curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, sess);
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);

	curl_easy_setopt(curl, CURLOPT_READFUNCTION, curl_read_cb);
	curl_easy_setopt(curl, CURLOPT_READDATA, sess);

	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, curl_write_cb);
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, sess);

	curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, curl_header_cb);
	curl_easy_setopt(curl, CURLOPT_HEADERDATA, sess);

	if (sess->token[0] == 0)
	{ 
		snprintf(buf, BUFFER_LEN, "%s://%s:%s/%s", 
			url_get_protocol(sess->tunnel),
			url_get_host(sess->tunnel),
			url_get_port(sess->tunnel),
			sess->clientid
		);
	}
	else
	{
		snprintf(buf, BUFFER_LEN, "%s://%s:%s/%s?%s", 
			url_get_protocol(sess->tunnel),
			url_get_host(sess->tunnel),
			url_get_port(sess->tunnel),
			sess->clientid,
			sess->token
		);
	}

	curl_easy_setopt(curl, CURLOPT_URL, buf);
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);

	res = curl_easy_perform(curl);
	if (res != CURLE_OK)
	{
		verboseWARN(&logger, "(%d) %s\n", res, curl_easy_strerror(res));
		if_sleep(10*1000);
	}

	curl_easy_cleanup(curl);
	curl_slist_free_all(chunk);

	return(res);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_FIRST(TUNNEL_SESSION *sess, void *user_data)
{
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_WAIT(TUNNEL_SESSION *sess, void *user_data)
{
	// TODO - criptografia
	char *purge[] = {"PORT","ADMINS","TUNNEL",NULL};

	for (int i = 0 ; purge[i] != NULL ; i++)
		json_object_remove(sess->info, purge[i]);

	json_object_add(sess->info, "version", json_string_new(siin_version()));

	sess->up_data = (uint8_t *) json_serialize(sess->info);
	sess->up_len = strlen((char *) sess->up_data);
	sess->up_pos = 0;

	tunnel_session_free_down(sess);

	if (tunnel_post(sess))
		sess->state = TUNNEL_SESSION_STATE_ERROR;

	tunnel_session_free_up(sess);

	if ((sess->down_len > 0) && (sess->type == TUNNEL_SESSION_WS))
		sess->state = TUNNEL_SESSION_STATE_WS;
	else
		sess->state = TUNNEL_SESSION_STATE_END;

	return(sess->state);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_WS(TUNNEL_SESSION *sess, void *user_data)
{
	char *ws = if_getopt_getValue(configs, "WS");
	size_t bufsize, packlen = 0x10000;
	int len, sent = 0;

	int sock = openTCP("127.0.0.1", sess->port, 5);
	if (sock < 1)
	{
		sess->state = TUNNEL_SESSION_STATE_ERROR;
		return(sess->state);
	}

	URL *wsurl = url_new(ws);
	char buf[PATH_LEN];
	len = snprintf(buf, PATH_LEN, "POST /%s HTTP/1.1\r\n", url_get_path(wsurl));
	url_free(wsurl);

	sent = send_bytes(sock, buf, len, 1);

	for (sess->down_pos = 0 ; (sess->down_len - sess->down_pos) > 0 ; sess->down_pos += sent)
	{
		if ((sess->down_len - sess->down_pos) > packlen)
			len = packlen;
		else
			len = sess->down_len - sess->down_pos;

		sent = send_bytes(sock, sess->down_data + sess->down_pos, len, 1);
		if (sent > 0)
		{
			sess->start = time(NULL);
			continue;
		}

		if (sent == 0)
		{
			if ((time(NULL) - sess->start) > 60)
			{
				sess->state = TUNNEL_SESSION_STATE_ERROR;
				goto tunnel_session_WS_err;
			}
		}
		else
		{
			sess->state = TUNNEL_SESSION_STATE_ERROR;
			goto tunnel_session_WS_err;
		}
	}

	bufsize = packlen;
	sess->up_data = if_malloc(bufsize);
	sess->up_len = 0;
	sess->up_pos = 0;

	for (sent = 0 ;	sent >= 0 ; sess->up_len += sent)
	{
		if ((bufsize - sess->up_len) < packlen)
		{
			bufsize *= 2;
			sess->up_data = realloc(sess->up_data, bufsize);
		}

		sent = read_timeout(sock, sess->up_data + sess->up_len, (int) packlen, 1);
		if (sent > 0)
		{
			sess->start = time(NULL);
			continue;
		}
		else if (sent == 0)
		{
			if ((time(NULL) - sess->start) > sess->timeout)
			{
				break;
			}
		}

		if (sent < 0)
			break;
	}

	sess->state = TUNNEL_SESSION_STATE_RETURN;

tunnel_session_WS_err:

	if_closesocket(sock);

	return(sess->state);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_RETURN(TUNNEL_SESSION *sess, void *user_data)
{
	// TODO - criptografia
	tunnel_post(sess);

	tunnel_session_free_up(sess);
	tunnel_session_free_down(sess);

	sess->state = TUNNEL_SESSION_STATE_END;

	return(sess->state);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_END(TUNNEL_SESSION *sess, void *user_data)
{
	// TODO - loga retorno
	tunnel_session_free_up(sess);
	tunnel_session_free_down(sess);
	sess->state = TUNNEL_SESSION_STATE_WAIT;
	return(sess->state);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION_STATE tunnel_session_ERROR(TUNNEL_SESSION *sess, void *user_data)
{
	tunnel_session_END(sess, user_data);
	return(sess->state);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void tunnel_session_free_down(TUNNEL_SESSION *sess)
{
	if_free(sess->down_data);
	sess->down_data = NULL;
	sess->down_len = 0;
	sess->down_pos = 0;
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_session_free_up(TUNNEL_SESSION *sess)
{
	if_free(sess->up_data);
	sess->up_data = NULL;
	sess->up_len = 0;
	sess->up_pos = 0;
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_SESSION * tunnel_session_new()
{
	char *clientid = if_getopt_getValue(configs, "CLIENTID");
	char *tunnel = if_getopt_getValue(configs, "TUNNEL");
	char *ws = if_getopt_getValue(configs, "WS");
	TUNNEL_SESSION *sess = NULL;
	JSON_VALUE *info = NULL;
	char buf[PATH_LEN];
	URL *url = url_new(ws);

	if (url == NULL)
	{
		verboseFATAL(&logger, "Parametro 'WS' invalido: '%s'\n", ws);
		return(NULL);
	}

	sess = if_malloc(sizeof(TUNNEL_SESSION));

	snprintf(sess->port, PORT_LEN, "%s", url_get_port(url));
	url_free(url);

	info = json_object_new(1);

	for (int i = 0 ; configs[i].long_opt != NULL ; i++)
	{
		char *key = configs[i].long_opt;
		char *val = if_getopt_getValue(configs, key);
		json_object_add(info, key, json_string_new(val));
	}

	siin_machine_info(info);
	sess->info = info;
	sess->timeout = 5;	// 5 segundos

	snprintf(buf, PATH_LEN, "%s/%s", json_object_get_string(info, "macaddress"), clientid);
	sess->clientid = if_strdup(buf);

	sess->type = TUNNEL_SESSION_ERROR;
	sess->state = TUNNEL_SESSION_STATE_WAIT;

	int i = 0;
	sess->func[i++] = tunnel_session_FIRST;
	sess->func[i++] = tunnel_session_WAIT;
	sess->func[i++] = tunnel_session_WS;
	sess->func[i++] = tunnel_session_RETURN;
	sess->func[i++] = tunnel_session_ERROR;
	sess->func[i++] = tunnel_session_END;

	snprintf(buf, PATH_LEN, "%s%s", tunnel, clientid);
	verboseINFO(&logger, "URL: '%s'\n", buf);
	sess->tunnel = url_new(buf);

	return(sess);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_session_free(TUNNEL_SESSION *sess)
{
	if (sess == NULL)
		return;

	if_free(sess->clientid);
	url_free(sess->tunnel);
	json_value_free(sess->info);
	if_free(sess);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void * tunnel_session_run(THREAD_STATE *state, int id, void *data)
{
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) data;
	if (sess == NULL)
		return(NULL);

	while ((*state == TH_ALIVE) && (sess->state < TUNNEL_SESSION_STATE_END))
	{
		verboseINFO(&logger, "Sessao: %lX (%d)\n", (uint64_t) sess, sess->state);

		sess->start = time(NULL);
		if (sess->func[sess->state] == NULL)
			sess->state++;
		else
			sess->func[sess->state](sess, NULL);
	}

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void * tunnel_th_event(IF_THREAD_LIST *list, IF_THREAD_EVENT *ev)
{
	int n = if_slist_length(list->first);
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) ev->data;

	switch (ev->event)
	{
		case IF_EVENT_START: fprintf(stdout, "Total: %d  -  EV: start (%X)\n", n, ev->id); break;
		case IF_EVENT_FREE: 
			fprintf(stdout, "Total: %d  -  EV: FREE (%X)\n", n, ev->id);
			tunnel_session_free(sess);
			break;

		case IF_EVENT_WAIT: break;
		case IF_EVENT_IDLE: if_sleep(1); break;
		case IF_EVENT_END: if_sleep(1000); break;
		default: break;
	}

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_thread_iter(IF_THREAD *th, void *user_data)
{
	TUNNEL_SESSION *sess = (TUNNEL_SESSION *) th->data;
	void **ctx = (void **) user_data;
	int *waiting = (int *) ctx[0];

	if (sess == NULL)
		printf("NULL\n\n\n");

	if (sess->state == TUNNEL_SESSION_STATE_WAIT)
		*waiting += 1;

	//verboseDEBUG(&logger, "(%lX) Waiting: %ld segs\n", th->id, time(NULL) - sess->start);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// Implementacao Modulo WEB //////////////////////////////////////////// //
// /////////////////////////////////////////////////////////////////// //
int if_modweb_init(MODSIIN *mod)
{
	char buf[PATH_LEN];
	int n;

	verbose(stdout, "Le configuracao: %s.\n", TUNNEL_INI);
	n = if_getopt_ini(TUNNEL_INI, configs);
	if (n < 1)
		verbose(stderr, "Gera arquivo: %s\n", TUNNEL_INI);

	snprintf(logger.source, sizeof(logger.source), "%s", MODULE);
	logger.verbosity = atoi(if_getopt_getValue(configs, "VERBOSITY"));

	verbose(stderr, "Sincroniza configuracoes do %s com: '%s'\n", MODULE, CONFIG_INI);
	for (int i = 0 ; configs[i].long_opt != NULL ; i++)
	{
		if ((strcmp(configs[i].long_opt, "TUNNEL") == 0) && (configs[i].arg[0] == 0))
			configs[i].arg = TUNNEL_SERVER; 

		n = if_getopt_key_in_file(CONFIG_INI, configs[i].long_opt, buf, PATH_LEN);
		if (n > 0)
		{
			if_getopt_setValue(configs, configs[i].long_opt, if_strdup(buf));
			verboseINFO(&logger, "%s: '%s' (%s)\n", configs[i].long_opt, configs[i].arg, CONFIG_INI);
		}
		else if (n < 0)
			verboseERROR(&logger, "Falha ao ler '%s' em: %s\n", configs[i].long_opt, CONFIG_INI);
		else
			verboseINFO(&logger, "%s: '%s' (%s)\n", configs[i].long_opt, configs[i].arg, TUNNEL_INI);
	}

	if_getopt_save(TUNNEL_INI, configs);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int if_modweb_finalize(MODSIIN *mod)
{
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
void * if_modweb_run(THREAD_STATE *state, int id, void *user_data)
{
	TUNNEL_SESSION *sess = tunnel_session_new();
	IF_THREAD_LIST *list;

	list = if_thread_list_new(tunnel_th_event);
	
	while (*state == TH_ALIVE)
	{
		int waiting = 0;
		void *ctx[] = {&waiting};
		if_thread_iter(list, tunnel_thread_iter, ctx);

		// TODO - implementar redirecionamento
		if (waiting == 0)
		{
			sess = tunnel_session_new();
			if_thread_add(list, tunnel_session_run, (intptr_t) sess, sess);
		}

		if_sleep(1000);
	}

	if_thread_list_finalize(list);

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
// FIM - Implementacao Modulo WEB ////////////////////////////////////// //



#ifdef DEBUG

#define DESCRIPT 	"Cliente Tunnel"

// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char *argv[])
{
	THREAD_STATE state = TH_ALIVE;

	if (argc < 2)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n\t$ %s -\n\n", argv[0]);
		return(1);
	}

	if_modweb_init(NULL);
	if_modweb_run(&state, 1, NULL);

	return(0);
} 
// ///////////////////////////////////////////////////////////////////// //
#endif

