/** @file siin.h Servico de Integracao Inteligente em Nuvem 
\ingroup 

\note -----
\author Felipe Peressoni Waltrick	felipe@ifractal.com.br
\version 1.1
**/

#ifndef SIIN_H
#define SIIN_H

#include <ifractal.h>

#define MACADDRESS_LEN		15
#define MAC_LEN			6

typedef struct _ICONNECTION	ICONNECTION;

/**
Callback da solicitação em background.

@return 0 para continuar - 1 interromper.
**/
typedef int (*CONN_EXECUTE_CALLBACK)(
	ICONNECTION *, 
	int total_upload, 
	int pos_upload, 
	int total_download, 
	int pos_download, 
	void *user_data);


/**
 * Contexto de conexao
**/
struct _ICONNECTION
{
	URL *url;			/*!< URL */
	char host[HOST_LEN];		/*!< HOST/IP do sistema */
	char port[PORT_LEN];		/*!< PORTA do sistema */
	char path[PATH_LEN];		/*!< PATH do cliente */
	JSON_VALUE *params;		/*!< Campos submetidos via POST */
	IF_THREAD_LIST *th_list;	/*!< Controle de threads do contexto */
	void *db_handle;
	char *content;
	int content_length;

	CONN_EXECUTE_CALLBACK cb;
	int total_upload;
	int pos_upload;
	int total_download;
	int pos_download;
	void *user_data;
};

/**
 * Contexto de dados
**/
typedef struct
{
	JSON_VALUE *data;		/*!< Dados em formato JSON */
	int qty_fields;			/*!< Quantidade de campos por registro */
	int qty_records;		/*!< Quantidade de registros */
	char *fields[PATH_LEN];		/*!< Nomes dos campos */
	JSON_VALUE *field_row;		/*!< Nomes dos campos */
	JSON_VALUE *current_row;
} IRESULTSET;

/**
 * Contexto de dados CNAB
**/
typedef struct
{
	JSON_VALUE *data;		/*!< Dados do CNAB em formato JSON */
	IRESULTSET *records;		/*!< Contexto de dados */
	int qty_global_fields;		/*!< Quantidade de campos globais (header) */
	char *globals[PATH_LEN];	/*!< Nome dos campos globais */
	JSON_VALUE *response;		/*!< Retorno do envio. */
	void *th_params[2];
	IF_THREAD *th;
} ICNAB;

ICNAB * cnab_new(JSON_VALUE *result);

ICONNECTION * db_connect(_IN char *path);
int db_close(_INOUT ICONNECTION *);
IRESULTSET * db_query(_IN ICONNECTION *conn, _IN char *query, _OUT int *affected);

char * siin_getDate(_IN time_t unixtime, _INOUT char *buf, int buflen);
char * siin_getTime(_IN time_t unixtime, _INOUT char *buf, int buflen);

typedef void (*SIIN_NET_IFACE)(char *iface, char *mac, char *ip, void *);
int siin_get_ifaces(SIIN_NET_IFACE, void *user_data);

char * siin_get_local_ip(_IN char *iface, _INOUT char *ip, size_t ip_len);
char * siin_mac(_INOUT char *buf, int buflen);
void siin_machine_info(_INOUT JSON_VALUE *info);
char * siin_hexlog(_IN unsigned char *buf, _IN int len);

unsigned char * siin_findBytes(
	_IN unsigned char *bytes, 
	_IN size_t byteslen,
	_IN unsigned char *buf, 
	_IN size_t buflen);

typedef int (*SIIN_READ_CALLBACK)(unsigned char *buf, size_t len, void *user_data);
int siin_findBytesInStream(unsigned char *bytes, size_t len, SIIN_READ_CALLBACK, void *user_data);

/**
Retorna o conteudo a partir da chave.

@param properties Vetor de strings
@param key
@return Conteudo do campo.
**/
char * siin_getParam(_IN char *properties[], _IN char *key);

/**
Altera o conteudo do campo a partir da chave.
Se a chave nao existir, esta funcao nao tem efeito.

@param properties Vetor de strings
@param key
@param value
@return indice do campo alterado no vetor.
**/
int siin_setParam(_IN char *properties[], _IN char *key, _IN char *value);


/**
Retorna versao da biblioteca.

@return Versao.
**/
char * siin_version();


/**
Instancia um contexto de conexao.
 
@param url do servidor WEB.
@return Nova instancia ICONNECTION.
**/
ICONNECTION * conn_create_url(char *url);

/**
Instancia um contexto de conexao.
 
@param host IP/nome do servidor WEB. (default: ifractal.srv.br) 
@param port WebPort
@param path Identificacao do sistema.
@return Nova instancia ICONNECTION.
**/
ICONNECTION * conn_create(char *host, char *port, char *path);

/**
Adiciona parametros na conexao.

@param conn Ponteiro do objeto ICONNECTION.
@param key Nome do parametro.
@param value Valor do parametro.
**/
void conn_addParam(ICONNECTION *conn, char *key, char *value);

/**
Envia um CNAB diretamente. Bloqueia o programa ate o fim do envio.

@param conn Ponteiro do objeto ICONNECTION.
@param cnab Ponteiro do objeto ICNAB.
@return 0 = Sucesso.
**/
int conn_sendCNAB(ICONNECTION *conn, ICNAB *cnab);

/**
Troca dados com sistema.<BR>
Funcao de conveniencia para redes que nao permitem conexoes entrantes.

@param conn Ponteiro do objeto ICONNECTION.
@param rs Ponteiro de um objeto IRESULTSET.
@return Objeto IRESULTSET.
**/
IRESULTSET * conn_sendData(ICONNECTION *conn, IRESULTSET *rs);

/**
Recupera dados do sistema.

@param conn Ponteiro do objeto ICONNECTION.
@return Objeto IRESULTSET.

Exemplo:
\code
	conn = conn_create(NULL, NULL, "clienteX");
	conn_addParam(conn, "login", user);
	conn_addParam(conn, "senha", pass);
	conn_addParam(conn, "pag", "relatorio_detalhe_envio");

	result = conn_getData(conn);
	if (result_count(result) < 0)
		fprintf(stderr, "ERRO MSG: %s\n", result_getErrParam(result, "msg"));
	else 
		show_result(result);

	result_close(result);
	conn_close(conn);
\endcode
**/
IRESULTSET * conn_getData(ICONNECTION *conn);


IRESULTSET * conn_getData_timeout(ICONNECTION *, int seconds);
IRESULTSET * conn_sendData_timeout(ICONNECTION *, IRESULTSET *, int seconds);


/**
Recupera dados do sistema.

@param conn Ponteiro do objeto ICONNECTION.
@return Conteudo busca. Nao é necessário liberar (free) a memória.

Exemplo:
\code
	conn = conn_create(NULL, NULL, "clienteX");
	conn_addParam(conn, "nome", "XYZ");

	conteudo = conn_execute(conn);
	conn_close(conn);
\endcode
**/
char * conn_execute(ICONNECTION *);


/**
Faz solicitação http (POST) em background.

@param conn Ponteiro do objeto ICONNECTION.
@param cb callback.
@param user_data Dados para o callback.
@return 0 = OK, 1 = Falha
**/
int conn_execute_background(ICONNECTION *, CONN_EXECUTE_CALLBACK, void *user_data);

/**
Recupera status da solicitação em background.

@param conn Ponteiro do objeto ICONNECTION.
@param total_upload Total a ser enviado.
@param pos_upload Subtotal enviado.
@param total_download Total a ser recebido.
@param pos_download Subtotal recebido.
@return 0 = em andamento, Diferente de 0 - não existe processo em background.
**/
int conn_background_state(
	_IN ICONNECTION *, 
	_OUT int *total_upload, 
	_OUT int *pos_upload, 
	_OUT int *total_download, 
	_OUT int *pos_download);


/**
Recupera tamanho (em bytes) da ultima resposta solicitada por conn_execute.

@param conn Ponteiro do objeto ICONNECTION.
@return tamanho em bytes
**/
int conn_get_content_length(ICONNECTION *);


/**
Recupera dados a partir de CSV.

@param conn Ponteiro do objeto ICONNECTION.
@return Objeto IRESULTSET.
**/
IRESULTSET * conn_getData_CSV(ICONNECTION *conn, char separador);


/**
Envia um CNAB em background (thread). Utilizar a funcao 'cnab_getSendingState' para saber o estado do envio.

@param conn Ponteiro do objeto ICONNECTION.
@param cnab Ponteiro do objeto ICNAB.
@return 0 = Sucesso.
**/
int conn_sendCNAB_background(ICONNECTION *conn, ICNAB *cnab);

/**
Retorna situacao do envio. 

@param conn Ponteiro do objeto ICONNECTION.
@param cnab Ponteiro do objeto ICNAB.

@return Menor que 0 - Erro, 0 - Finalizado, Maior que 0 - em andamento.
**/
int cnab_getSendingState(ICONNECTION *conn, ICNAB *cnab);

/**
Recupera o valor de um parametro de retorno.

@param cnab Ponteiro do objeto ICNAB.
@param param_name Nome do parametro. 

@return Valor do Parametro. 
**/
char * cnab_getResponseParam(ICNAB *cnab, char *param_name);

/**
Recupera o valor de um parametro de retorno, para o caso de ocorrencia de erro. 

@param rs Ponteiro do objeto IRESULTSET.
@param param_name Nome do parametro. 

@return Valor do Parametro. 
**/
char * result_getErrParam(IRESULTSET *rs, char *param_name);

/**
Encerra um contexto de conexao. Quando nao utilizada, causa "memory leak".

@param conn Ponteiro do objeto ICONNECTION.
**/
void conn_close(ICONNECTION *conn);



/**
Retorna a quantidade de registros de um contexto de dados.

@param rs Ponteiro do objeto IRESULTSET.

@return Quantidade de registros.
**/
int result_count(IRESULTSET *rs);

/**
Retorna a quantidade de campos dos registros de um contexto de dados.

@param rs Ponteiro do objeto IRESULTSET.

@return Quantidade de campos por registro.
**/
int result_getFieldCount(IRESULTSET *rs);

/**
Retorna o nome do campo a partir do indice

@param rs Ponteiro do objeto IRESULTSET.
@param index Posicao do campo (a partir do indice 0)

@return Nome do campo.
**/
char * result_getFieldName(IRESULTSET *rs, int index);

/**
Retorna o indice do campo a partir do nome.

@param rs Ponteiro do objeto IRESULTSET.
@param field_name Nome do campo.

@return Posicao do campo.
**/
int result_getFieldIndex(IRESULTSET *rs, char *field_name);

/**
Retorna o conteudo do campo a partir do indice.

@param rs Ponteiro do objeto IRESULTSET.
@param index Indice do registro (iniciando em 0)
@param field_index Indice do campo (iniciando em 0)

@return Valor do campo.
**/
char * result_getValue(IRESULTSET *rs, int index, int field_index);

/**
Encerra um contexto de dados. Quando nao utilizada, causa "memory leak".

@param rs Ponteiro do objeto IRESULTSET.
**/
void result_close(IRESULTSET *rs);

/**
Inicia um contexto IRESULTSET a partir de um arquivo CSV.

@param csv_file Path/Nome do arquivo CSV. 
@param separador Separador utilizado no arquivo. 

@return Novo objeto IRESULTSET.
**/
IRESULTSET * result_importCSV(char *csv_file, char separador);

/**
Inicia um contexto de dados CNAB.

@param cnab_file Arquivo texto de posicoes fixas.
@param template_file Arquivo de definicoes do respectivo CNAB. (fornecido pela iFractal)

@return Novo objeto ICNAB.
**/
ICNAB * cnab_importFile(char *cnab_file, char *template_file);

/**
Adiciona campos de mailing ao contexto de dados CNAB.

@param mailing_file Arquivo texto CSV com, no minimo, os campos; ID;EMAIL;SMS
@param id Nome do campo chave.

@return Quantidade de registros localizados.
**/
int cnab_addMailing(_INOUT ICNAB *cnab, char *mailing_file, char separador, char *id);


/**
Encerra um contexto de dados CNAB. Quando nao utilizada, causa "memory leak".

@param cnab Ponteiro do objeto ICNAB.
**/
void cnab_close(ICNAB *cnab);

/**
Retorna somente os registros do CNAB, sem informacoes de "header".

@param cnab Ponteiro do objeto ICNAB.

@return IRESULTSET a partir dos registros do CNAB.
**/
IRESULTSET * cnab_getRecords(ICNAB *cnab);

/**
Retorna quantidade de campos globais (campos de header) do CNAB.

@param cnab Ponteiro do objeto ICNAB.

@return Quantidade de campos.
**/
int cnab_getGlobalCount(ICNAB *cnab);

/**
Retorna nome do campo global a partir do indice. 

@param cnab Ponteiro do objeto ICNAB.
@param index Indice do campo. 

@return Nome do campo. 
**/
char *cnab_getGlobalName(ICNAB *cnab, int index);

/**
Retorna conteudo do um campo global a partir do indice. 

@param cnab Ponteiro do objeto ICNAB.
@param index Indice do campo. 

@return Valor do campo. 
**/
char * cnab_getGlobalValue(ICNAB *cnab, int index);


/**
Inicializa um IRESULTSET vazio.

@return Objeto IRESULTSET.

Exemplo:
\code
	char *fields[] = {"id", "nome", "descricao", NULL};
	IRESULTSET *rs, *rs_ret;

	rs = result_new();

	for (i = 0 ; fields[i] != NULL ; i++)
		result_column_add(rs, fields[i]);

	result_row_add(rs);
	result_row_set_value(rs, "id", "11");
	result_row_set_value(rs, "nome", "Tabela 3");
	result_row_set_value(rs, "descricao", "Plano A");

	rs_ret = conn_sendData(conn, rs);

	result_close(rs);
	result_close(rs_ret);
\endcode
**/
IRESULTSET * result_new();


/**
Inicializa um IRESULTSET a partir de um JSON.

@param json Ponteiro do objeto JSON_VALUE.
@return Objeto IRESULTSET.
**/
IRESULTSET * result_new_json(JSON_VALUE *json);


/**
Adiciona uma coluna ao RESULTSET.

@param rs Ponteiro do objeto IRESULTSET.
@param column_name Nome da coluna.
**/
void result_column_add(_IN IRESULTSET *rs, _IN char *column_name);

/**
Adiciona um registro ao RESULTSET.

@param rs Ponteiro do objeto IRESULTSET.
**/
void result_row_add(_IN IRESULTSET *rs);

/**
Adiciona valor a um campo do registro corrente.

@param rs Ponteiro do objeto IRESULTSET.
@param column_name Nome da coluna.
@param value Valor.
**/
void result_row_set_value(_IN IRESULTSET *rs, char * column_name, char * value);

#endif
