#include <ifractal.h>

#ifndef WIN32
#include <errno.h>

extern int errno;
#endif

// //////////////////////////////////////////////////////////////////////
_PRIVATE int ping_recv(int pingsock, BROADCAST_RESPONSE_CALLBACK cb, void *user_data)
{
	struct sockaddr_in from;
	int n, size = BUFFER_LEN;
	char packet[size];
	int r;

	socklen_t fromlen = sizeof(from);

#ifdef WIN32
	n = recvfrom(pingsock, (char *) &packet, sizeof(packet), MSG_PEEK, (struct sockaddr *) &from, &fromlen);
	if (n == 0)
		return(n);

	if (n > 0)
		n = recvfrom(pingsock, (char *) &packet, sizeof(packet), 0, (struct sockaddr *) &from, &fromlen);
#else
	n = recvfrom(pingsock, packet, sizeof(packet), MSG_DONTWAIT, (struct sockaddr *) &from, &fromlen);
#endif
	if (n < 1)
		return(0);

	if (cb == NULL)
		return(0);

	r = cb(inet_ntoa(from.sin_addr), 0, (BROADCAST_DATA *) packet, n, user_data);
	if (r)
		return(r);

	return(0);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PRIVATE void ping_recv_timeout(int pingsock, BROADCAST_RESPONSE_CALLBACK cb, void *user_data, int timeout)
{
	time_t start = time(NULL);
	int r;

	do
	{
		r = ping_recv(pingsock, cb, user_data);
		if (r)
			return;

		if_sleep(100);
	}
	while ((time(NULL) - start) < timeout);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PRIVATE int ping_open()
{
	int sock;

#ifdef WIN32
	WSADATA WSData;
	if ( WSAStartup(MAKEWORD(2,2), &WSData) )
	{
		verbose(stderr, "ERROR: You need Winsock.\n");
		return(-1);
	}

	
	if ((sock = WSASocket(AF_INET, SOCK_RAW, IPPROTO_ICMP, 0, 0, 0)) == SOCKET_ERROR)
	{
		fprintf(stderr, "ERROR creating socket (%d)\n", WSAGetLastError());
	  	return(-3);
	}

	unsigned long int nonBlockingMode = 1;
	ioctlsocket(sock, FIONBIO, &nonBlockingMode);

#else
	//if ((sock = socket(AF_INET, SOCK_RAW, IPPROTO_RAW )) == -1)
	if ((sock = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP )) == -1)
	{
		fprintf(stderr, "ERROR creating socket\n");
	  	return(-3);
	}
#endif

	return(sock);
}
// //////////////////////////////////////////////////////////////////////
_PRIVATE int ping_send_in(int sock, struct sockaddr_in *dst, unsigned char *data, size_t datalen)
{
	struct
	{
		struct icmphdr icmp;
		//unsigned char data[datalen];
		unsigned char data[512];
	} packet;

	memset(&(packet.icmp), 0, sizeof(struct icmphdr));
	memset(packet.data, 0, sizeof(packet.data));
	memcpy(packet.data, data, sizeof(packet.data));

#ifndef WIN32
	// TODO - possibilidade de alterar o pacote ip
	/*
	//memcpy(&(packet.ip.ip_src.s_addr, hp->h_addr_list[0], hp->h_length);

	int on = 1;
	setsockopt(sock, IPPROTO_IP, IP_HDRINCL, &on, sizeof(on));

	ip->ip_v = 4;
	ip->ip_hl = sizeof(struct ip) / 4;
	ip->ip_tos = 0;
	ip->ip_len = htons(sizeof(buf));
	ip->ip_id = htons(31337);
	ip->ip_off = htons(0);
	ip->ip_ttl = 255;
	ip->ip_p = IPPROTO_ICMP;
	ip->ip_sum = 0;	// kernel handles this one
	*/
#endif

	packet.icmp.type = ICMP_ECHO;
	packet.icmp.code = 0;
	packet.icmp.checksum = in_cksum((uint16_t *) &packet, sizeof(packet));

#ifdef WIN32
	if (sendto(sock, (char *) &packet, sizeof(packet), 0, (struct sockaddr *) dst, sizeof(struct sockaddr_in)) < 0)
	{
		verbose(stderr, "ERROR sending packet.\n");
		return(-4);
	}
#else
	if (sendto(sock, &packet, sizeof(packet), 0, (struct sockaddr *) dst, sizeof(struct sockaddr_in)) < 0)
	{
		verbose(stderr, "ERROR sending packet: %d\n", errno);
		return(-4);
	}
#endif

	return(sock);
}
// //////////////////////////////////////////////////////////////////////
_PRIVATE int ping_send(struct sockaddr_in *dst, unsigned char *data, size_t datalen)
{
	int sock;

	sock = ping_open();
	if (sock >= 0)
		ping_send_in(sock, dst, data, datalen);

	return(sock);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int ping_perform_timeout(
	char *ip_dst, 
	unsigned char *data, 
	size_t datalen, 
	BROADCAST_RESPONSE_CALLBACK cb, 
	void *user_data,
	int timeout)
{
	struct sockaddr_in dst;
	struct hostent *hp;
	int sock;

	dst.sin_family = AF_INET;

	if ((dst.sin_addr.s_addr = inet_addr(ip_dst)) == -1)
	{
		if ((hp = gethostbyname(ip_dst)) == NULL)
		{
	  		fprintf(stderr, "ERROR looking up host %s\n", ip_dst);
	  		return(-2);
		} 
		else
			memcpy(&dst.sin_addr, hp->h_addr, hp->h_length);
	}
	sock = ping_send(&dst, data, datalen);
	if (sock < 0)
		return(sock);

	ping_recv_timeout(sock, cb, user_data, timeout);

	if_closesocket(sock);

	return(0);
}
// //////////////////////////////////////////////////////////////////////


#ifdef WIN32
// //////////////////////////////////////////////////////////////////////
_PUBLIC BROADCAST_CONTEXT * broadcast_icmp(
	BROADCAST_RESPONSE_CALLBACK cb, 
	void *user_data)
{
	INTERFACE_INFO InterfaceList[MAX_IFACES];
	unsigned char macs[MAX_IFACES][6];
	BROADCAST_CONTEXT *ctx;
	struct sockaddr_in *dst;
	int i, ifs, sock;
	unsigned char j;
	unsigned char data[] = "iFractal Diagnostico"; 
	size_t datalen = sizeof(data);

	memset(macs, 0, sizeof(macs));

	sock = ping_open();
	if (sock < 0)
		return(NULL);

	ifs = netraw_win_ifconfig(sock, InterfaceList, macs);
	for (i = 0; i < ifs ; i++)
	{
		dst = (struct sockaddr_in *) &(InterfaceList[i].iiAddress);

		if (((unsigned char *) &dst->sin_addr.s_addr)[0] == 127)
			continue;

		for (j = 1 ; j < 255 ; j++)
		{
			((unsigned char *) &dst->sin_addr.s_addr)[3] = j;
			ping_send_in(sock, dst, data, datalen);
			ping_recv(sock, cb, user_data);
		}
	}

	ctx = if_malloc(sizeof(BROADCAST_CONTEXT));
	memset(ctx, 0, sizeof(BROADCAST_CONTEXT));

	ctx->sock = sock;
	ctx->cb = cb;
	ctx->user_data = user_data;

	return(ctx);
}
// //////////////////////////////////////////////////////////////////////
#else
// //////////////////////////////////////////////////////////////////////
_PUBLIC BROADCAST_CONTEXT * broadcast_icmp(
	BROADCAST_RESPONSE_CALLBACK cb, 
	void *user_data)
{
	BROADCAST_CONTEXT *ctx;
	struct sockaddr_in *dst;
	struct ifconf ifconf;
	struct ifreq ifr[MAX_IFACES];
	unsigned char macs[MAX_IFACES][6];
	int n, i, ifs, sock;
	unsigned char j;
	unsigned char data[] = "iFractal Diagnostico"; 
	size_t datalen = sizeof(data);
	struct ifreq *ifr_ptr;

	ifconf.ifc_buf = (char *) ifr;
	ifconf.ifc_len = sizeof(ifr);

	memset(macs, 0, sizeof(macs));

	sock = ping_open();
	if (sock < 0)
		return(NULL);

	ifs = netraw_ifconfig(sock, &ifconf, ifr, macs);

	for (n = 0, i = 0, ifr_ptr = ifr ; i < ifs ; i++, ifr_ptr = (struct ifreq *)(n + (char *) ifr_ptr))
	{
#ifdef __MACH__
		n = sizeof(struct ifreq) - sizeof(struct sockaddr) + MAX(sizeof(struct sockaddr), ifr_ptr->ifr_addr.sa_len);

		dst = (struct sockaddr_in *) &(ifr_ptr->ifr_addr);
#else
		dst = (struct sockaddr_in *) &ifr[i].ifr_addr;
#endif
		if (dst->sin_family != AF_INET)
			continue;

		if (((unsigned char *) &dst->sin_addr.s_addr)[0] == 127)
			continue;

		//int k; 
		//printf("%s - %02X", ifr[i].ifr_name, macs[i][0]);
		//for (k = 1 ; k < 6 ; k++)
		//	printf(":%02X", macs[i][k]);
		//printf("\n");

		for (j = 1 ; j < 255 ; j++)
		{
			((unsigned char *) &dst->sin_addr.s_addr)[3] = j;
			ping_send_in(sock, dst, data, datalen);
			//printf("x.x.x.%d\n", ((unsigned char *) &dst->sin_addr.s_addr)[3]);
		}
	}

	ctx = if_malloc(sizeof(BROADCAST_CONTEXT));
	memset(ctx, 0, sizeof(BROADCAST_CONTEXT));

	ctx->sock = sock;
	ctx->cb = cb;
	ctx->user_data = user_data;

	return(ctx);
}
// //////////////////////////////////////////////////////////////////////
#endif



#ifdef STANDALONE
// //////////////////////////////////////////////////////////////////////
int ping_response(char *ip, int port, BROADCAST_DATA *data, size_t datalen, void *user_data)
{
	struct packet
	{
		struct ip ip;
		struct icmphdr icmp;
	} *pkt;
	int i;

	pkt = (struct packet *) data;

	fprintf(stdout, "%s (%ld) ICMP - type: %d,  code: %d\n", ip, datalen, pkt->icmp.type, pkt->icmp.code);

/*
	fprintf(stdout, "DATA: --> ");
	for (i = 0 ; i < datalen ; i++)
		fprintf(stdout, "%02X ", data->data[i]);
	fprintf(stdout, "<--\n\n");

	fflush(stdout);
*/
	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int main (int argc, char *argv[])
{
	//time_t start = time(NULL);
	char str[6000] = "1234567";

	if (argc < 2) 
	{
		printf("Uso:\n\tshell$ %s <dest>\n", argv[0]);
		return(1);
	}

	ping_perform_timeout(argv[1], (unsigned char *) str, sizeof(str), ping_response, NULL, 10);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
#endif


