#include <ifractal.h>

#include <jni.h>


// //////////////////////////////////////////////////////////////////////
JavaVM * if_loadJVM(char *classpath_in, char *libpath_in, char *stack_in)
{
	char *classpath = ".";
	char *libpath = "";
	char *stack = "1";

	if (classpath_in != NULL)
		classpath = classpath_in;

	if (libpath_in != NULL)
		libpath = libpath_in;

	if (stack_in != NULL)
		stack = stack_in;

	char *opts[] = {
		"-Djava.compiler=%s", "NONE",
		"-Djava.class.path=%s", classpath,
		"-Djava.library.path=%s", libpath,
		"-Xss%sm", stack,
		"-verbose:%s", "gc",
		"-XX:-CreateMinidumpOnCrash", NULL,
		NULL, NULL
	};
	int opts_qty = ((sizeof(opts) / sizeof(char *)) / 2) - 1;

	JavaVM *jvm;
	JNIEnv *env;
	JavaVMInitArgs vm_args;
	JavaVMOption options[opts_qty];
	char opt[opts_qty][BUFFER_LEN];
	long result;
	int i, ptr;

	verbose(stdout, "Parametros JVM\n");
	for (i = 0, ptr = 0 ; opts[ptr] != NULL ; i++, ptr += 2)
	{
		snprintf(opt[i], sizeof(opt[i]), opts[ptr], opts[ptr + 1]);
		options[i].optionString = opt[i];
		verbose(stdout, "\t%s\n", opt[i]);
	}

	vm_args.version = JNI_VERSION_1_2;
	vm_args.options = options;
	vm_args.nOptions = opts_qty;
	vm_args.ignoreUnrecognized = JNI_FALSE;

	verbose(stdout, "Inicia JVM\n");
	result = JNI_CreateJavaVM(&jvm, (void **) &env, &vm_args);
	if(result == JNI_ERR) 
	{
		verbose(stderr, "Erro ao tentar carregar JVM.\n");
		return(NULL);
	}

	return(jvm);
}
// //////////////////////////////////////////////////////////////////////
void if_unloadJVM(JavaVM *jvm)
{
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
jobjectArray if_JVM_getStringArray(JNIEnv *env, int argc, char *argv[])
{
	if (argc < 1)
		return(NULL);

	jobjectArray objarray = (*env)->NewObjectArray(
		env, 
		argc, 
		(*env)->FindClass(env, "java/lang/String"), 
		NULL);

	for (int i = 0 ; i < argc ; i++)
		(*env)->SetObjectArrayElement(env, objarray, i, (*env)->NewStringUTF(env, argv[i]));

	return(objarray);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int if_JVM_attach(
	_IN JavaVM *jvm,
	_IN intptr_t obj,
	_OUT JNIEnv **env,
	_OUT jclass *klass)
{
	int r = 0;

	jint res = (*jvm)->AttachCurrentThread(jvm, (void **) env, NULL);
	if (res < 0)
	{
		verbose(stderr, "Thread attach failed\n");
		return(-1);
	}

	*klass = (**env)->GetObjectClass(*env, (jobject) obj);
	if (*klass == NULL) 
	{
		verbose(stderr, "Classe nao encontrada.\n");
		(**env)->ExceptionDescribe(*env);
		r = -2;
	}
	(**env)->ExceptionClear(*env);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
int if_JVM_attach_and_find_class(
	_IN JavaVM *jvm,
	_IN char *classname,
	_OUT JNIEnv **env,
	_OUT jclass *klass)
{
	int r = 0;

	jint res = (*jvm)->AttachCurrentThread(jvm, (void **) env, NULL);
	if (res < 0)
	{
		verbose(stderr, "Thread attach failed\n");
		return(-1);
	}

	*klass = (**env)->FindClass(*env, classname);
	if (*klass == NULL) 
	{
		verbose(stderr, "Classe '%s' nao encontrada.\n", classname);
		(**env)->ExceptionDescribe(*env);
		r = -2;
	}
	(**env)->ExceptionClear(*env);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
void if_JVM_dettach(_IN JavaVM *jvm)
{
	(*jvm)->DetachCurrentThread(jvm);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
int if_JVM_callStaticInt64Method(
	_IN JavaVM *jvm,
	_IN char *classname, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT int64_t *returnValue)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	int r = 0;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach_and_find_class(jvm, classname, &env, &klass);
	if (r)
	{
		r = -2;
		goto callStaticInt64_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*returnValue = 0;

	mid = (*env)->GetStaticMethodID(env, klass, meth, "([Ljava/lang/String;)J");
	if (mid)
	{
		*returnValue = (*env)->CallStaticLongMethod(env, klass, mid, objarray);
		r = 0;
	}
	else
	{
		verbose(stderr, "metodo '%s.%s' nao encontrado.\n", classname, meth);
		r = -3;
		goto callStaticInt64_error;
	}

callStaticInt64_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
int if_JVM_callStaticStringMethod(
	_IN JavaVM *jvm, 
	_IN char *classname, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT char **returnValue)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	jobject ret;
	int r;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach_and_find_class(jvm, classname, &env, &klass);
	if (r)
	{
		r = -2;
		goto callStaticString_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*returnValue = NULL;

	mid = (*env)->GetStaticMethodID(env, klass, meth, "([Ljava/lang/String;)Ljava/lang/String;");
	if (mid)
	{
		r = 0;
		ret = (*env)->CallStaticObjectMethod(env, klass, mid, objarray);
	}
	else
	{
		verbose(stderr, "metodo '%s.%s' nao encontrado.\n", classname, meth);
		r = -3;
		goto callStaticString_error;
	}

	if (ret == NULL)
	{
		r = -4;
		goto callStaticString_error;
	}

	const char *str;
	if (ret != NULL)
	{
		str = (*env)->GetStringUTFChars(env, ret, 0);
		*returnValue = if_strdup((char *) str);
		(*env)->ReleaseStringUTFChars(env, ret, str);
	}

callStaticString_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
int if_JVM_callStaticBytesMethod(
	_IN JavaVM *jvm, 
	_IN char *classname, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT uint8_t **data,
	_OUT size_t *datalen)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	jobject ret;
	int r;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach_and_find_class(jvm, classname, &env, &klass);
	if (r)
	{
		r = -2;
		goto callStaticBytes_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*data = NULL;

	mid = (*env)->GetStaticMethodID(env, klass, meth, "([Ljava/lang/String;)[B");
	if (mid)
	{
		r = 0;
		ret = (*env)->CallStaticObjectMethod(env, klass, mid, objarray);
	}
	else
	{
		verbose(stderr, "metodo '%s.%s' nao encontrado.\n", classname, meth);
		r = -3;
		goto callStaticBytes_error;
	}

	if (ret == NULL)
	{
		r = -4;
		goto callStaticBytes_error;
	}

	*datalen = (*env)->GetArrayLength(env, ret);
	jbyte *arr = (*env)->GetByteArrayElements(env, ret, 0);
	*data = if_malloc(*datalen + 1);
	(*data)[*datalen] = 0;
	for (int i = 0 ; i < *datalen ; i++)
		(*data)[i] = arr[i];
	(*env)->ReleaseByteArrayElements(env, ret, arr, 0);

callStaticBytes_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int if_JVM_callStaticObjectMethod(
	_IN JavaVM *jvm, 
	_IN char *classname, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT intptr_t *obj)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	jobject ret;
	int r;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach_and_find_class(jvm, classname, &env, &klass);
	if (r)
	{
		r = -2;
		goto callStaticObject_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*obj = 0;

	mid = (*env)->GetStaticMethodID(env, klass, meth, "([Ljava/lang/String;)Ljava/lang/Object;");
	if (mid)
	{
		r = 0;
		ret = (*env)->CallStaticObjectMethod(env, klass, mid, objarray);
	}
	else
	{
		verbose(stderr, "metodo '%s.%s' nao encontrado.\n", classname, meth);
		r = -3;
		goto callStaticObject_error;
	}

	if (ret == NULL)
	{
		r = -4;
		goto callStaticObject_error;
	}

	*obj = (intptr_t) ret;

callStaticObject_error:
	if_JVM_dettach(jvm);

	return(r);

}
// //////////////////////////////////////////////////////////////////////
// ([Ljava/lang/String;)J
int if_JVM_callInt64Method(
	_IN JavaVM *jvm, 
	_IN intptr_t obj, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT int64_t *returnValue)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	int r = 0;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach(jvm, obj, &env, &klass);
	if (r)
	{
		r = -2;
		goto callInt64_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*returnValue = 0;

	mid = (*env)->GetMethodID(env, klass, meth, "([Ljava/lang/String;)J");
	if (mid)
	{
		*returnValue = (*env)->CallLongMethod(env, (jobject) obj, mid, objarray);
		r = 0;
	}
	else
	{
		verbose(stderr, "metodo '%s' nao encontrado.\n", meth);
		r = -3;
		goto callInt64_error;
	}

callInt64_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
// ([Ljava/lang/String;)Ljava/lang/String;
int if_JVM_callStringMethod(
	_IN JavaVM *jvm, 
	_IN intptr_t obj, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT char **returnValue)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	jobject ret;
	int r;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach(jvm, obj, &env, &klass);
	if (r)
	{
		r = -2;
		goto callString_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*returnValue = NULL;

	mid = (*env)->GetMethodID(env, klass, meth, "([Ljava/lang/String;)Ljava/lang/String;");
	if (mid)
	{
		r = 0;
		ret = (*env)->CallObjectMethod(env, (jobject) obj, mid, objarray);
	}
	else
	{
		verbose(stderr, "metodo '%s' nao encontrado.\n", meth);
		r = -3;
		goto callString_error;
	}

	if (ret == NULL)
	{
		r = -4;
		goto callString_error;
	}

	const char *str;
	if (ret != NULL)
	{
		str = (*env)->GetStringUTFChars(env, ret, 0);
		*returnValue = if_strdup((char *) str);
		(*env)->ReleaseStringUTFChars(env, ret, str);
	}

callString_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////
// ([Ljava/lang/String;)[B
int if_JVM_callBytesMethod(
	_IN JavaVM *jvm, 
	_IN intptr_t obj, 
	_IN char *meth, 
	_IN size_t argc,
	_IN char *argv[], 
	_OUT uint8_t **data,
	_OUT size_t *datalen)
{
	jobjectArray objarray;
	jmethodID mid;
	jclass klass;
	JNIEnv *env;
	jobject ret;
	int r;

	if (argc < 0)
		return(-1);

	r = if_JVM_attach(jvm, obj, &env, &klass);
	if (r)
	{
		r = -2;
		goto callBytes_error;
	}

	objarray = if_JVM_getStringArray(env, argc, argv);
	*data = NULL;

	mid = (*env)->GetMethodID(env, klass, meth, "([Ljava/lang/String;)[B");
	if (mid)
	{
		r = 0;
		ret = (*env)->CallObjectMethod(env, (jobject) obj, mid, objarray);
	}
	else
	{
		verbose(stderr, "metodo '%s' nao encontrado.\n", meth);
		r = -3;
		goto callBytes_error;
	}

	if (ret == NULL)
	{
		r = -4;
		goto callBytes_error;
	}

	*datalen = (*env)->GetArrayLength(env, ret);
	jbyte *arr = (*env)->GetByteArrayElements(env, ret, 0);
	*data = if_malloc(*datalen + 1);
	(*data)[*datalen] = 0;
	for (int i = 0 ; i < *datalen ; i++)
		(*data)[i] = arr[i];
	(*env)->ReleaseByteArrayElements(env, ret, arr, 0);

callBytes_error:
	if_JVM_dettach(jvm);

	return(r);
}
// //////////////////////////////////////////////////////////////////////

#ifdef JAVA_DEBUG

// //////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
	char *classpath = ".";
	char *libpath = "";
	int64_t returnValue = 0;
	char *retstr = NULL;
	uint8_t *data = NULL;
	size_t datalen = 0;

	if (argc < 3)
	{
		fprintf(stderr, "Uso:\n\t$ %s <CLASS> <METHOD>\n", argv[0]);
		return(1);
	}

	JavaVM *jvm = if_loadJVM(classpath, libpath, NULL);
/*
	int r = if_JVM_callStaticStringMethod(jvm, argv[1], argv[2], argc, argv, &retstr);
	if (!r)
	{
		fprintf(stdout, "returnValue: -->%s<-- (%d)\n", retstr, r);
		if_free(retstr);
	}

	r = if_JVM_callStaticInt64Method(jvm, argv[1], argv[2], argc, argv, &returnValue);
	if (!r)
		fprintf(stdout, "returnValue: %ld (%d)\n", returnValue, r);

	r = if_JVM_callStaticBytesMethod(jvm, argv[1], argv[2], argc, argv, &data, &datalen);
	if (!r)
	{
		fprintf(stdout, "len: %ld (%d)\n", datalen, r);
		for (int k = 0 ; k < datalen ; k++)
			fprintf(stdout, "%02X ", data[k]);
		fprintf(stdout, "\n");
		if_free(data);
	}
*/

	intptr_t obj = 0;
	int r = if_JVM_callStaticObjectMethod(jvm, argv[1], argv[2], argc, argv, &obj);
	if ((r) || (obj == 0))
	{
		fprintf(stderr, "Falha ao tentar obter instancia (%d)\n", r);
		return(2);
	}

	r = if_JVM_callInt64Method(jvm, obj, "init", argc, argv, &returnValue);
	if (!r)
		fprintf(stdout, "returnValue: %ld (%d)\n", returnValue, r);

	r = if_JVM_callStringMethod(jvm, obj, "testStr", argc, argv, &retstr);
	if (!r)
	{
		fprintf(stdout, "returnValue: -->%s<-- (%d)\n", retstr, r);
		if_free(retstr);
	}

	r = if_JVM_callBytesMethod(jvm, obj, "testBytes", argc, argv, &data, &datalen);
	if (!r)
	{
		fprintf(stdout, "len: %ld (%d)\n", datalen, r);
		for (int k = 0 ; k < datalen ; k++)
			fprintf(stdout, "%02X ", data[k]);
		fprintf(stdout, "\n");
		if_free(data);
	}

	if_unloadJVM(jvm);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

#endif

