#include <ifractal.h>
#include <ifdevice.h>
#include <henry8x.h>


// ///////////////////////////////////////////////////////// //
_PRIVATE char * henry8x_getErrMsg(int err)
{
	switch (err)
	{
		case -1: return("timeout");
		case -2: return("header invalido");
		case -3: return("falha sincronizacao");
		case -4: return("falha ao tentar ler/enviar pacote");
		case -5: return("falha ao tentar ler trailer");
		case -6: return("falha checksum");
		case -7: return("falha end byte");
		case 0: return("OK");
		case 1: return("Nao ha dados");
		case 10: return("Comando desconhecido");
		case 11: return("Tamanho do pacote invalido");
		case 12: return("Parametros informados invalidos");
		case 13: return("Erro de checksum");
		case 14: return("Tamanho dos parametros invalidos");
		case 15: return("Numero da mensagem invalido");
		case 16: return("Start Byte invalido");
		case 17: return("Erro para receber pacote");
		case 20: return("Nao ha empregador cadastrado");
		case 21: return("Nao ha usuarios cadastrados");
		case 22: return("Usuario nao cadastrado");
		case 23: return("Usuario ja cadastrado");
		case 24: return("Limite de cadastro de usuarios atingido");
		case 25: return("Equipamento nao possui biometria");
		case 26: return("Index biometrico nao encontrado");
		case 27: return("Limite de cadastro de digitais atingido");
		case 28: return("Equipamento nao possui eventos");
		case 29: return("Erro na manipulacao de biometrias");
		case 30: return("Documento do empregador invalido");
		case 31: return("Tipo do documento do empregador invalido");
		case 32: return("Ip invalido");
		case 33: return("Tipo de operacao do usuario invalida");
		case 34: return("Pis do empregado invalido");
		case 35: return("Cei do empregador invalido");
		case 36: return("Referencia do empregado invalida");
		case 37: return("Referencia de cartao de usuario invalida");
		case 38: return("Referencia de sequencia de acesso e invalida");
		case 39: return("Referencia de horario invalida");
		case 40: return("Referencia de periodo invalida");
		case 43: return("Erro ao gravar dados");
		case 44: return("Erro ao ler dados");
		case 50: return("Sem registros a partir desse NSR.");
		case 61: return("Matricula ja existe");
		case 62: return("Pis ja existe");
		case 63: return("Opcao invalida");
		case 64: return("Matricula nao existe");
		case 65: return("PIS nao existe");
		case 113: return("NSR inexistente.");
		case 180: return("Erro de tipo de operacao de Horario ");
		case 181: return("Erro de tipo de operacao de Periodo ");
		case 182: return("Erro de tipo de operacao de Escala");
		case 240: return("Registro nao foi encontrado");
		case 241: return("Registro ja existe");
	}

	return("Reservado");
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PRIVATE char * henry8x_getDescEvento(int ev)
{
	switch (ev)
	{
		case 0: return("Aguardando validacao: Equipamento esta aguardando uma resposta do servidor para o evento enviado.");
		case 1: return("Liberado: Indica que o evento foi validado e liberado. Em caso de catraca libera ambos os lados.");
		case 2: return("Giro de catraca: obsoleta, indicado agora em 81.");
		case 3: return("Liberado por Master: Indica que a liberacao foi feita atraves de cartao master.");
		case 4: return("Liberacao online: Indica que foi enviada uma liberacao atraves do webserver ou aplicativo.");
		case 5: return("Libera entrada: Libera indicando direcao (catraca).");
		case 6: return("Libera saida: Libera indicando direcao (catraca).");
		case 20: return("Mensagem: Indica ao equipamento que deve ser exibida uma mensagem no display de acordo com os parametros informados.");
		case 21: return("Status: Reservado para implementacao futura.");
		case 22: return("Tamper interno acionado: Tamper interno foi acionado (equipamento aberto indevidamente).");
		case 23: return("Tamper externo acionado: Tamper externo (parede) foi acionado.");
		case 30: return("Negado: Acesso negado.");
		case 31: return("Negado por horario: Fora do horario cadastrado. Pode ser indicado pelo servidor ou pelo equipamento no modo offline.");
		case 32: return("Negado por acesso: Usuario nao possui os privilegios para acessar o equipamento. Pode ser indicado pelo servidor ou pelo equipamento no modo offline.");
		case 33: return("Negado por senha: A senha digitada nao confere com a cadastrada.");
		case 34: return("Negado por sequencia de acesso: A sequencia de acesso nao foi correta.");
		case 35: return("Negado por validade do cartao: Cartao esta fora da validade.");
		case 36: return("Negado por sentido: Sentido esta errado (entrar sem sair, sair sem entrar)");
		case 40: return("Senha de Emergencia: Usuario utilizou senha de emergencia.");
		case 80: return("Aguardando Giro: Indica que o equipamento recebeu a validacao, e agora esta aguardando o usuario transpor a catraca.");
		case 81: return("Giro de catraca: Usuario transpôs a catraca.");
		case 82: return("Desistencia de giro: Usuario desistiu de transpor a catraca.");
		case 83: return("Catraca forcada: Catraca esta sendo forcada ou usada incorretamente.");
		case 84: return("Catraca em giro: Catraca esta em giro. Usuario esta parado no meio.");
		case 85: return("Timeout de cofre: Usuario nao depositou o cartao no cofre.");
		case 88: return("Cofre obstruido: ha um cartao invalido depositado no cofre,");
		case 90: return("Desistencia: Reservado");
		case 91: return("Timeout online: Servidor nao respondeu a requisicao do equipamento no tempo definido.");
		case 95: return("Botao: Botao foi pressionado.");
		case 100: return("Alarme: Entrada digital atrelada a alarme foi acionada.");
		case 120: return("Porta aberta: Porta esta aberta.");
		case 121: return("Porta Forcada: Porta esta sendo forcada.");
		case 122: return("Porta Arrombada: Porta foi arrombada.");
	}
	
	return("Evento indeterminado");
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
HENRY_TYPE henry8x_getType(IFDEVICE *dev)
{
	char *model = json_object_get_string(dev->config, "MODELO");
	char *modelos_REP[] = {"Primme SF", "", NULL};	// "" = Prisma
	char *modelos_ACESSO[] = {"ACESSO", "primmeAcesso", NULL};
	char *modelos_ARGOS[] = {"ARGOS", NULL};
	struct
	{
		char **model;
		HENRY_TYPE type;
	} models[] = {
		{modelos_ACESSO, HENRY_ACESSO},
		{modelos_ARGOS, HENRY_ARGOS},
		{modelos_REP, HENRY_PONTO},
		{NULL, 0}
	};
	int i, j;

	for (i = 0 ; models[i].model != NULL ; i++)
	{
		for (j = 0 ; models[i].model[j] != NULL ; j++)
			if (strcasecmp(models[i].model[j], model) == 0)
				return(models[i].type);
	}

	verboseFATAL(&(dev->log), "Modelo nao identificado: '%s'.\n", model);

	return(HENRY_UNKNOWN);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
int henry8x_addResult(JSON_VALUE *res, JSON_VALUE *reg, char oper, char *resp, int err)
{
	char *format = "%s (%c) - %s | %s";
	char msg[PATH_LEN];
	JSON_VALUE *evt = json_object_new(1);
	char *cod = json_object_get_string(reg, "cod");
	char *codigo = json_object_get_string(reg, "codigo");
	char *nome = json_object_get_string(reg, "nome");
	int n = 0;
	time_t now = time(NULL);
	struct tm *dt;

	dt = localtime(&now);
	snprintf(msg, sizeof(msg), "%04d-%02d-%02d %02d:%02d:%02d", 
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday, dt->tm_hour, dt->tm_min, dt->tm_sec);

	json_object_add(evt, "datahora", json_string_new(msg));

	if (resp == NULL)
	{
		n = snprintf(msg, sizeof(msg), format, codigo, oper, nome, "");
	}
	else
	{
		n = snprintf(msg, sizeof(msg), format, codigo, oper, nome, resp);
		if_free(resp);
	}

	json_object_add(evt, "cod", json_string_new(cod));
	json_object_add(evt, "codigo", json_string_new(codigo));
	json_object_add(evt, "status", json_string_new(msg));
	json_object_add(evt, "erro", json_string_new(henry8x_getErrMsg(err)));
	json_object_add(evt, "cod_error", json_integer_new(err));
	json_array_add(res, evt);

	return(n);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
void henry8x_convert_date(char *from, char *to)
{
	int i = 0;

	// 2019-01-19 00:00:00
	to[i++] = from[8];
	to[i++] = from[9];
	to[i++] = '/';
	to[i++] = from[5];
	to[i++] = from[6];
	to[i++] = '/';
	to[i++] = from[0];
	to[i++] = from[1];
	to[i++] = from[2];
	to[i++] = from[3];
	to[i++] = ' ';
	to[i++] = from[11];
	to[i++] = from[12];
	to[i++] = ':';
	to[i++] = from[14];
	to[i++] = from[15];
	to[i++] = ':';
	to[i++] = from[17];
	to[i++] = from[18];
	to[i++] = 0;

	if (from[1] == '1')
	{
		to[7] = '0';
		to[8] = '4';
		to[9] = '0';
	}

}
// ///////////////////////////////////////////////////////// //
int henry8x_get_horarios_iter(JSON_VALUE *jhorario, void *user_data)
{
	void **ctx = (void **) user_data;
	int *qty = (int *) ctx[0];
	char *buf = (char *) ctx[1];
	char **ref = (char **) ctx[2];
	char *p = *ref;

	*ref += snprintf(p, PATH_LEN - (*ref - buf), "%s{", json_get_string(jhorario));
	*qty += 1;

	return(0);
}
// ///////////////////////////////////////////////////////// //
int henry8x_sendUser_acesso(IFDEVICE *dev, JSON_VALUE *user, JSON_VALUE *res, char *codigo, char *ver_bio, JSON_VALUE *jcartao)
{
	char *header = "00+ECAR+00+1+";
	char *req_format = "%s%c[%s[%s[%s[%s[%d[0%d[%s[[[[[2[%d[%s[0[[0";
	char dthr_inicio[] = "14/01/2019 15:40:00";
	char dthr_fim[] = "29/08/2040 15:08:00";
	HENRY_COD_ACESSO cod_acesso = HENRY_COD_ACESSO_LIBERADO;
	HENRY_TYPE type = henry8x_getType(dev);
	unsigned char *resp = NULL;
	char buf[PATH_LEN];
	char cmd = 'I';
	int err, n;
	char *tipo = json_object_get_string(jcartao, "tipo");
	char *tipo_operacao = json_object_get_string(user, "tipo");
	int cod_tipo = 1;
	char *nro = json_object_get_string(jcartao, "nro");
	char *ativo = json_object_get_string(jcartao, "ativo");
	char *devolucao = json_object_get_string(jcartao, "devolucao");
	int qty_horarios = 0;

	if ((strncasecmp(tipo_operacao, "excluir", 7) == 0) || (ativo[0] == '0') || (devolucao[0] == 't')) 
		cmd = 'E';

	if (json_object_find(jcartao, "hrinicio") != NULL)
	{
		henry8x_convert_date(json_object_get_string(jcartao, "hrinicio"), dthr_inicio);
		cod_acesso = HENRY_COD_ACESSO_VALIDADE;	
	}

	if (json_object_find(jcartao, "hrfim") != NULL)
	{
		henry8x_convert_date(json_object_get_string(jcartao, "hrfim"), dthr_fim);
		cod_acesso = HENRY_COD_ACESSO_VALIDADE;
	}

	if (strncasecmp(tipo, "func", 4) != 0)
	{
		ver_bio = "0";
		cod_tipo = 1;
	}

	if (strncasecmp(tipo, "visi", 4) == 0)
		cod_tipo = 4;
	else if (strncasecmp(tipo, "prov", 4) == 0)
		cod_tipo = 5;

	if (ver_bio[0] == '0')
		verboseDEBUG(&(dev->log), "Cartao: %s (%s) - Verifica biometria DESABILITADA.\n", nro, tipo);
	else
		verboseDEBUG(&(dev->log), "Cartao: %s (%s) - Verifica biometria HABILITADA.\n", nro, tipo);

	char horarios_str[PATH_LEN] = "";
	char *ptr = horarios_str;
	JSON_VALUE *jhorarios = json_object_find(user, "horarios");
	void *ctx[] = {&qty_horarios, horarios_str, &ptr};
	if ((jhorarios != NULL) && (json_get_type(jhorarios) == JSON_ARRAY_TYPE) && (type != HENRY_ARGOS))
	{
		cod_acesso = HENRY_COD_ACESSO_HORARIO;
		json_array_iter(jhorarios, henry8x_get_horarios_iter, ctx);
	}

	n = snprintf(buf, PATH_LEN, req_format, 
		header,
		cmd, 
		codigo,
		nro,
		dthr_inicio,
		dthr_fim,
		cod_acesso,
		cod_tipo,
		ver_bio,
		qty_horarios,
		horarios_str);

	n = henry8x_send(dev, (unsigned char *) buf, n);
	resp = NULL;
	n = henry8x_recv(dev, (unsigned char **) &resp, &err);
	if ((cmd != 'E') || (json_array_length(res) < 1))
		henry8x_addResult(res, user, cmd, (char *) resp, err);

	return(n);
}
// ///////////////////////////////////////////////////////// //
int henry8x_sendUsers_acesso_iter(JSON_VALUE *user, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	JSON_VALUE *res = (JSON_VALUE *) params[1];

	JSON_VALUE *jnro_cartao = json_object_find(user, "nro_cartao");
	JSON_VALUE *elemento;
	int i;
	int qtd;
	char *codigo = json_object_get_string(user, "pis");
	if (codigo[0] == 0)
		codigo = json_object_get_string(user, "codigo");

	char *ver_bio = json_object_get_string(user, "verificar_biometria");

	if ((jnro_cartao != NULL) && (json_get_type(jnro_cartao) == JSON_ARRAY_TYPE) && (json_array_length(jnro_cartao) > 0))
	{ 
		qtd = json_array_length(jnro_cartao);
		for (i = 0 ; i < qtd ; i++)
		{
			elemento = json_array_index(jnro_cartao, i); 
			henry8x_sendUser_acesso(dev, user, res, codigo, ver_bio, elemento);
		}
	}
	else
	{
		codigo = json_object_get_string(user, "cod");
		henry8x_sendUser_acesso(dev, user, res, codigo, "0", user);
	}

	return(0);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE int henry8x_sendUser_REP(
	_IN IFDEVICE *dev, 
	_INOUT JSON_VALUE *res,
	_IN JSON_VALUE *reg,
	_IN char **params, 
	_IN char cmd) 
{
	char header[] = "00+EU+00+1+";
	char req_format[] = "%s%c[%s[%s[%s[1[%s";
	char req[PATH_LEN];
	int n, err;
	char *codigo = json_get_list_param(params, "pis");
	
	if (codigo[0] == 0)
		codigo = json_get_list_param(params, "codigo");

	if (codigo[0] == 0)
		return(0);

	n = snprintf(req, PATH_LEN, req_format, 
		header,
		cmd, 
		codigo,
		json_get_list_param(params, "nome"),
		json_get_list_param(params, "verificar_biometria"),
		json_get_list_param(params, "cracha"));

	err = henry8x_send(dev, (unsigned char *) req, n);
	if (n < 0)
		henry8x_addResult(res, reg, cmd, NULL, err);

	return(n);
}
// ///////////////////////////////////////////////////////// //
_CALLBACK int henry8x_sendUsers_iter(_IN JSON_VALUE *user, void *user_data)
{
	void **data = (void **) user_data;
	IFDEVICE *dev = data[0];
	JSON_VALUE *res = (JSON_VALUE *) data[1];
	char *params[] = {
		"codigo","",
		"cracha","",
		"verificar_biometria","", 
		"nome","",
		"pis","",
		"tipo","",
		NULL,NULL
	};
	char *resp = NULL;
	char cmd = 'I'; 	// I(1) inclusao, A(2) - alteracao, E(3) - exclusao
	int n, err; 

	json_make_list_param(user, params);

	char *tipo = json_get_list_param(params, "tipo");

	if ((strncmp(tipo, "excluir", 7) == 0) || (strncmp(tipo, "devolver", 8) == 0))
		cmd = 'E';
	
	n = henry8x_sendUser_REP(dev, res, user, params, cmd);
	if (n < 0)
		return(n);

	resp = NULL;
	n = henry8x_recv(dev, (unsigned char **) &resp, &err);
	henry8x_addResult(res, user, cmd, resp, err);

	return(0);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
int henry8x_get_empregador(IFDEVICE *dev)
{
	char *fields[] = {"_", "cnpj", "cei", "razao", "local", NULL};
	char req_header[] = "00+RE+00+";
	unsigned char *pack;
	int l = sizeof(fields) / sizeof(char *);
	int size, n, i, r = 0;
	char *tk[l];

	if ((n = henry8x_send((IFDEVICE *) dev, (unsigned char *) req_header, strlen(req_header))) < 0)
		return(-1);

	if ((size = henry8x_recv((IFDEVICE *) dev, &pack, NULL)) <= 0)
		return(size);

	n = tokenizer(']', (char *) pack, tk, l);
	if (n < (l - 1))
	{
		r = -2;
		goto henry8x_get_empregador_end;
	}

	for (i = 1 ; fields[i] != NULL ; i++)
		json_object_add(dev->config, fields[i], json_string_new(tk[i]));

henry8x_get_empregador_end:
	if_free(pack);

	return(r);
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
_PRIVATE int henry8x_process_coleta_REP_check(char *line)
{
	if (strlen(line) < 34)
		return(0);

	if (line[9] != '3')
		return(0);

	return(1);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE void henry8x_coleta_json_acesso(IFDEVICE *dev, JSON_VALUE *obj, char *line)
{
	JSON_VALUE *json_config;
	int canal_cofre;
	char datahora[PATH_LEN];
	char *tk[CSV8x_LEN];
	char nro_cartao[20];
	char *p;
	int n;

	n = tokenizer(' ', line, tk, CSV8x_LEN);
	if (n < CSV8x_HORA)
		return;

	snprintf(datahora, PATH_LEN, "%04d-%02d-%02d %s", 
		atoi(tk[CSV8x_DATA] + 6),
		atoi(tk[CSV8x_DATA] + 3),
		atoi(tk[CSV8x_DATA]),
		tk[CSV8x_HORA]);

	// Elimina ZEROS a esquerda
	for (p = tk[CSV8x_MATRICULA] ; (*p == '0') && (*p != 0) ; p++)
		;

	if (atoi(p) == 0)
	{
		snprintf(nro_cartao, sizeof(nro_cartao), "0");
		json_object_add(obj, "nro_cartao", json_string_new("0"));
	}
	else
	{
		snprintf(nro_cartao, sizeof(nro_cartao), "%d", atoi(p));
		json_object_add(obj, "nro_cartao", json_string_new(nro_cartao));
	}

	int canal = atoi(tk[CSV8x_LEITOR]);
	json_object_add(obj, "datahora", json_string_new(datahora));
	json_object_add(obj, "canal", json_integer_new(canal));

	json_config = json_object_find(dev->config, "json_config"); 
	if ((json_config != NULL) && (json_object_find(json_config, "canal_cofre") != NULL))
	{
		canal_cofre = json_object_get_int(json_config, "canal_cofre");
		if (canal_cofre == canal)
			json_object_add(obj, "cofre", json_integer_new(1));
	}

	char *sentido;

	if (atoi(tk[CSV8x_SENTIDO]) == HENRY8x_ENTRADA)
	{
		sentido = "E";
	}
	else
	{
		sentido = "S";
	}
	json_object_add(obj, "sentido", json_string_new(sentido));

	char *efetivado = "1";
	int cod_evento = atoi(tk[CSV8x_COD_EVT]);

	if ((cod_evento >= 30) && (cod_evento < 40))
	{
		json_object_add(obj, "descricao", json_string_new(henry8x_getDescEvento(cod_evento)));
		sentido = "B";
		efetivado = "0";
	}

	if (cod_evento == 82)
		efetivado = "0";

	json_object_add(obj, "efetivado", json_string_new(efetivado));
	json_object_add(obj, "sentido", json_string_new(sentido));


	verboseINFO(&(dev->log), "%s|%s|%s|%s|%d\n", datahora, nro_cartao, sentido, efetivado, canal);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE JSON_VALUE * henry8x_coleta_json_REP(JSON_VALUE *obj, char *line)
{
	char datahora[PATH_LEN];

	json_object_add(obj, "afd", json_string_new(line));

	snprintf(datahora, PATH_LEN, "20%c%c-%c%c-%c%c %c%c:%c%c:00", 
		line[16], line[17], 	// ano
		line[12], line[13], 	// mes
		line[10], line[11], 	// dia
		line[18], line[19], 	// hora
		line[20], line[21] 	// min
		);

	json_object_add(obj, "datahora", json_string_new(datahora));
	json_object_add(obj, "pis", json_string_new(line + 23));

	return(obj);
}
// ///////////////////////////////////////////////////////// //
_CALLBACK int henry8x_coleta_json_iter(_IN JSON_VALUE *jline, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	JSON_VALUE *joffs = (JSON_VALUE *) params[1];
	int *nsr = (int *) params[2];
	char *line = json_get_string(jline);
	JSON_VALUE *obj = NULL;
	int acesso = 1;
	int n, aux;

	if (line[0] == 0)
		return(0);

	for (aux = 0, n = 0 ; n < 9 ; n++)
	{
		if ((line[n] < '0') || (line[n] > '9'))
			break;

		aux *= 10;
		aux += line[n] - '0';
	}

	if (line[n] == ' ')
	{
		acesso = 1;
	}
	else if (line[n + 1] == ' ')
	{
		aux *= 10;
		aux += line[n] - '0';
		acesso = 1;
	}
	else
		acesso = 0;

	if ((aux > 0) && (aux < 2000000))
		*nsr = aux;
	else
		return(0);

	obj = json_object_new(1);
	json_object_add(obj, "nsr", json_integer_new(aux));

	if (acesso)
	{
		// Coleta Acesso
		henry8x_coleta_json_acesso(dev, obj, line);
	}
	else
	{
		if (henry8x_process_coleta_REP_check(line))
		{
			// Registro tipo 3 - marcacao
			henry8x_coleta_json_REP(obj, line);
		}
		else
		{
			// Envia todos os eventos do AFD
			json_object_add(obj, "afd", json_string_new(line));
		}
	}

	json_array_add(joffs, obj);

	return(0);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE JSON_VALUE * henry8x_coleta_json(_INOUT IFDEVICE *reader, _IN JSON_VALUE *jlines, _INOUT intptr_t *nsr)
{
	JSON_VALUE *joffs = json_array_new(2);
	void *params[] = {reader, joffs, nsr};

	json_array_iter(jlines, henry8x_coleta_json_iter, params);

	return(joffs);
}
// ///////////////////////////////////////////////////////// //
_PRIVATE JSON_VALUE * henry8x_coleta_in(IFDEVICE *reader, char *cmd, intptr_t *nsr)
{
	unsigned char *resp = NULL, *p;
	JSON_VALUE *jlines, *joffs = NULL;
	int n, err, nlines;
	char **lines;
	char *tk[5];

	if ((n = henry8x_send(reader, (unsigned char *) cmd, strlen(cmd))) < 0)
		return(NULL);

	n = henry8x_recv(reader, &resp, &err);
	if ((n == 0) || (resp == NULL))
		return(json_array_new(1));

	// Substitui separador de linha e campo
	for (p = resp ; *p != 0 ; p++)
	{
		if (*p == ']')
			*p = '\n';

		if (*p == '[')
			*p = ' ';
	}

	nlines = parser_get_lines((char *) resp, &lines);

	jlines = json_array_new(2);

	if (tokenizer('+', lines[0], tk, 5) > 4)
		json_array_add(jlines, json_string_new(tk[4]));

	for (n = 1 ; n < nlines ; n++)
		json_array_add(jlines, json_string_new(lines[n]));

	if_free(lines);
	if_free(resp);

	joffs = henry8x_coleta_json(reader, jlines, nsr);
	json_value_free(jlines);

	return(joffs);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_CALLBACK int henry8x_parseBio(_IN JSON_VALUE *user, unsigned char *resp, size_t len)
{
	JSON_VALUE *templates = json_array_new(1);
	JSON_VALUE *temp;
	unsigned char *p;
	char *aux;
	int n = 0;

	json_object_add(user, "templates", templates);
	for (p = resp ; (p - resp) < len ; p++)
	{
		if (*p != '{')
			continue;

		temp = json_object_new(1);
		json_array_add(templates, temp);
		json_object_add(temp, "vendor", json_string_new("suprema"));

		// TODO - verificar: hexa / base64
		n++;
		aux = bin2hex(p + 1, SUPREMA_BIO_LEN);
		json_object_add(temp, "template", json_string_new(aux));
		if_free(aux);
		p += SUPREMA_BIO_LEN;
	}

	return(n);
}
// ///////////////////////////////////////////////////////// //
_CALLBACK int henry8x_getBio_iter(_IN JSON_VALUE *user, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	int *q = (int *) params[1];
	char format[] = "%s%s";
	char header[] = "00+RD+000+D]";
	char req[sizeof(header) + 20];
	unsigned char *resp = NULL;
	char *cracha = json_object_get_string(user, "cracha");
	int n, err;

	if (cracha[0] == 0)
		return(0);

	snprintf(req, sizeof(req), format, header, cracha);
	if ((n = henry8x_send(dev, (unsigned char *) req, strlen(req))) < 0)
		return(-1);

	n = henry8x_recv(dev, &resp, &err);
	if ((n <= 0) || (resp == NULL))
		return(0);

	if (n > SUPREMA_BIO_LEN)
	{
		henry8x_parseBio(user, resp, n);
		*q += 1;
	}

	if_free(resp);

	return(0);
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
int henry8x_sendBio_pack(int cracha, JSON_VALUE *templates, unsigned char *req, int max)
{
	int qty_temp = json_array_length(templates);
	char header[] = "00+ED+000+D";
	char *hextemp, *vendor;
	unsigned char *p = req;
	unsigned char *temp;
	JSON_VALUE *aux;
	int i;

	p += snprintf((char *) p, max, "%s]%020d}%d}", header, cracha, qty_temp);
	for (i = 0 ; i < qty_temp ; i++)
	{
		aux = json_array_index(templates, i);

		vendor = json_object_get_string(aux, "vendor");
		if ((vendor[0] != 0) && (strcmp(vendor, "suprema") != 0))
			continue;

		// TODO - verificar: hexa / base64
		hextemp = json_object_get_string(aux, "template");
		p += snprintf((char *) p, max, "%d{", i + 1);
		temp = hex2bin(hextemp);
		memcpy(p, temp, SUPREMA_BIO_LEN);
		if_free(temp);
		p += SUPREMA_BIO_LEN;
	}

	return(p - req);
}
// ///////////////////////////////////////////////////////// //
int henry8x_deleteBio(IFDEVICE *dev, JSON_VALUE *user, JSON_VALUE *res, int cracha)
{
	char header[] = "00+ED+000+E]%020d";
	char req[PATH_LEN];
	char *resp = NULL;
	int n, err;

	n = snprintf(req, sizeof(req), header, cracha);
	if ((n = henry8x_send(dev, (unsigned char *) req, n)) < 0)
		return(n);

	n = henry8x_recv(dev, (unsigned char **) &resp, &err);
	if_free(resp);

	return(n);
}
// ///////////////////////////////////////////////////////// //
_CALLBACK int henry8x_sendBio_iter(_IN JSON_VALUE *user, void *user_data)
{
	void **params = (void **) user_data;
	IFDEVICE *dev = (IFDEVICE *) params[0];
	JSON_VALUE *res = (JSON_VALUE *) params[1];
	JSON_VALUE *templates = json_object_find(user, "templates");
	int cracha = atoi(json_object_get_string(user, "cracha"));
	unsigned char *req = NULL;
	unsigned char *resp = NULL;
	int n, err, qty_temp, len;

	if (cracha == 0)
		return(0);

	if ((templates == NULL) || (json_get_type(templates) != JSON_ARRAY_TYPE))
		return(0);

	qty_temp = json_array_length(templates);
	if (qty_temp < 1)
		return(0);

	len = 30 + (qty_temp * (SUPREMA_BIO_LEN + 20));
	req = if_malloc(len);

	len = henry8x_sendBio_pack(cracha, templates, req, len);
	if (len < SUPREMA_BIO_LEN)
	{
		henry8x_addResult(res, user, 'I', if_strdup("Nenhuma biometria incluida."), 0);
		if_free(req);
		return(0);
	}

	if_sleep(500);
	henry8x_deleteBio(dev, user, res, cracha);
	if_sleep(500);

	if ((n = henry8x_send(dev, (unsigned char *) req, len)) < 0)
	{
		if_free(req);
		return(-1);
	}

	if_free(req);

	n = henry8x_recv(dev, &resp, &err);
	henry8x_addResult(res, user, 'I', (char *) resp, err);

	return(0);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
int getUserBioQty_iter(JSON_VALUE *user, void *user_data)
{
	IFDEVICE *dev = (IFDEVICE *) user_data;
	char *cracha = json_object_get_string(user, "cracha");
	char header[] = "00+RD+000+Q}%s";
	char req[PATH_LEN];
	char *resp = NULL;
	char *tk[5];
	int n, err;

	n = snprintf(req, sizeof(req), header, cracha);
	if ((n = henry8x_send(dev, (unsigned char *) req, n)) < 0)
		return(n);

	n = henry8x_recv(dev, (unsigned char **) &resp, &err);
	if ((n < 1) || (resp == NULL))
		return(n);

	n = tokenizer('}', resp, tk, 4);
	if (n < 2)
		json_object_add(user, "bio_qty", json_integer_new(1));
	else
		json_object_add(user, "bio_qty", json_integer_new(atoi(tk[1])));

	if_free(resp);

	return(0);
}
// ///////////////////////////////////////////////////////// //
int getUserBioQty(IFDEVICE *dev, JSON_VALUE *users)
{
	return(json_array_iter(users, getUserBioQty_iter, dev));
}
// ///////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////
JSON_VALUE * henry8x_getInfo(IFDEVICE4J *dev, long sock)
{
	// "00+RC+00+GATEWAY]MASC_SUBREDE]MAC]NR_REP]VERSAO_PRODUTO]VERSAO_PROTOCOLO";
	IFDEVICE *reader = (IFDEVICE *) dev;
	char *params[] = {
		"GATEWAY",
		"MASC_SUBREDE",
		"MAC",
		"NR_REP",
		"MODELO",
		"VERSAO_PRODUTO",
		"VERSAO_PROTOCOLO",
		NULL};
	char req_header[] = "00+RC+000+";
	char req[PATH_LEN], *tk[52];
	char *p, *resp;
	int i, n, size;

	p = req;
	p += snprintf(p, PATH_LEN, "%s%s", req_header, params[0]);
	for (i = 1 ; params[i] != NULL ; i++)
		p += snprintf(p, PATH_LEN - (p - req), "]%s", params[i]);

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(NULL);

	if ((n = henry8x_send(reader, (unsigned char *) req, strlen(req))) < 0)
		goto henry8x_getInfo_err;

	if ((size = henry8x_recv(reader, (unsigned char **) &resp, NULL)) <= 0)
		goto henry8x_getInfo_err;

	for (n = 0 ; resp[n] != 0 ; n++)
	{
		if ((resp[n] == '[') || (resp[n] == ']'))
			resp[n] = '|';

		if ((resp[n] == '\r') || (resp[n] == '\n'))
			resp[n] = ' ';
	}

	n = tokenizer('|', resp + strlen(req_header), tk, 50);
	tk[n] = NULL;

	for (i = 0 ; (i + 1) < n ; i += 2)
		json_object_add(((IFDEVICE *) dev)->config, tk[i], json_string_new(tk[i + 1]));

	if_free(resp);

	if (henry8x_getType(dev) == HENRY_PONTO)
		henry8x_get_empregador((IFDEVICE *) dev);

henry8x_getInfo_err:

	henry8x_close((IFDEVICE *) dev);
	return(json_clone(((IFDEVICE *) dev)->config));
}
// ////////////////////////////////////////////////////////////////////////
time_t henry8x_getTime(IFDEVICE4J *dev, long sock)
{
	char req_datahora[] = "00+RH+00";
	int ano, mes, dia, hor, min, seg;
	unsigned char *pack;
	char *global_tk[6];
	char *tk[4], *p;
	int n, size;
	time_t ret;

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(0);

	if ((n = henry8x_send((IFDEVICE *) dev, (unsigned char *) req_datahora, strlen(req_datahora))) < 0)
		goto henry8x_getTime_err;

	if ((size = henry8x_recv((IFDEVICE *) dev, &pack, NULL)) <= 0)
		goto henry8x_getTime_err;

	n = tokenizer('+', (char *) pack, global_tk, 4);
	if (n < 4)
		goto henry8x_getTime_err;

	n = tokenizer(']', global_tk[3], tk, 4);
	if (n < 3)
		goto henry8x_getTime_err;

	p = tk[0];
	dia = atoi(p);
	p += 3;
	mes = atoi(p);
	p += 3;
	ano = atoi(p);

	p += 3;
	hor = atoi(p);
	p += 3;
	min = atoi(p);
	p += 3;
	seg = atoi(p);

	struct tm info;
	info.tm_year = 100 + ano;
	info.tm_mon = mes - 1;
	info.tm_mday = dia;
	info.tm_hour = hor;
	info.tm_min = min;
	info.tm_sec = seg;
	info.tm_isdst = -1;

	ret = mktime(&info);

	if_free(pack);

	henry8x_close((IFDEVICE *) dev);

	return(ret);

henry8x_getTime_err:

	henry8x_close((IFDEVICE *) dev);
	return(0);
}
// ////////////////////////////////////////////////////////////////////////
intptr_t henry8x_setTime(IFDEVICE4J *dev, intptr_t diff)
{
	char send_datahora[] = "00+EH+00+%02d/%02d/%02d %02d:%02d:%02d]00/00/00]00/00/00";
	unsigned char pack[PATH_LEN], *resp;
	struct tm *dt;
	int n, r = 0;
	time_t now;

	now = time(NULL);
	now += 3600 * diff;
	dt = localtime(&now);

	n = snprintf((char *) pack, PATH_LEN, send_datahora, 
		dt->tm_mday, 
		dt->tm_mon + 1, 
		dt->tm_year - 100, 
		dt->tm_hour, 
		dt->tm_min,
		dt->tm_sec);

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(0);

	if ((n = henry8x_send((IFDEVICE *) dev, pack, n)) < 0)
		goto henry8x_setTime_err;

	if ((n = henry8x_recv((IFDEVICE *) dev, &resp, NULL)) < 0)
		goto henry8x_setTime_err;

	if_free(resp);
	r = 1;

henry8x_setTime_err:
	henry8x_close((IFDEVICE *) dev);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////
JSON_VALUE * henry8x_getEvents(IFDEVICE4J *dev4j, intptr_t nsr)
{
	IFDEVICE *dev = (IFDEVICE *) dev4j;
	HENRY_TYPE type = henry8x_getType(dev4j);
	JSON_VALUE *eventos = NULL;
	char cmd[PATH_LEN] = "";
	int n = (int) nsr & 0xFFFFFFFF;

	switch (type)
	{
		case HENRY_ARGOS:
		case HENRY_ACESSO:
			snprintf((char *) cmd, PATH_LEN, "00+RR+00+T]50]%09d", n);
			break;

		case HENRY_PONTO:
			snprintf((char *) cmd, PATH_LEN, "00+RR+00+N]50]%d", n);
			break;

		case HENRY_UNKNOWN:
		case HENRY_TYPE_LEN:
			break;
	}

	if (cmd[0] == 0)
	{
		goto henry8x_getEvents_err;
	}

	if ((henry8x_connect(dev)) < 1)
		return(NULL);

	eventos = henry8x_coleta_in(dev, cmd, &nsr);

	henry8x_close(dev);

	return(eventos);

henry8x_getEvents_err:
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * henry8x_sendUsers(IFDEVICE4J *dev, JSON_VALUE *users)
{
	HENRY_TYPE type = henry8x_getType(dev);
	JSON_VALUE *res = json_array_new(1);
	void *params[] = {dev, res};

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(res);

	switch (type)
	{
		case HENRY_ARGOS:
			json_array_iter(users, henry8x_sendUsers_acesso_iter, params);
			break;

		case HENRY_ACESSO:
			json_array_iter(users, henry8x_sendUsers_acesso_iter, params);
			// TODO - em alguns equipamentos de acesso eh necessario executar "00+EU+00" junto com "00+ECAR+00"
			break;

		case HENRY_PONTO:
			json_array_iter(users, henry8x_sendUsers_iter, params);
			break;

		case HENRY_UNKNOWN:
		case HENRY_TYPE_LEN:
			break;
	}

	henry8x_close((IFDEVICE *) dev);

	return(res);
}
// ////////////////////////////////////////////////////////////////////////
JSON_VALUE * henry8x_getBio(IFDEVICE4J *dev, JSON_VALUE *users)
{
	int qty = 0;
	void *params[] = {dev, &qty};

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(0);

	json_array_iter(users, henry8x_getBio_iter, params);
	henry8x_close((IFDEVICE *) dev);

	return(json_clone(users));
}
// ////////////////////////////////////////////////////////////////////////
JSON_VALUE * henry8x_sendBio(IFDEVICE4J *dev, JSON_VALUE *users)
{
	JSON_VALUE *res = json_array_new(1);
	void *params[] = {dev, res};

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(res);

	json_array_iter(users, henry8x_sendBio_iter, params);
	henry8x_close((IFDEVICE *) dev);

	return(res);
}
// ////////////////////////////////////////////////////////////////////////
JSON_VALUE * henry8x_getUsers(IFDEVICE4J *dev, intptr_t none)
{
	char format[] = "%s%d}%d";
	char header[] = "00+RD+000+L]";
	char req[sizeof(header) + 10];
	JSON_VALUE *users = json_array_new(1);
	JSON_VALUE *user, *cartao, *nro_cartao;
	char *pack = NULL;
	int n, size, err, i, j, q;
	int qty = 50;
	char *tk[qty + 3];

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(users);

	for (i = 0 ; ; i += qty)
	{
		n = snprintf(req, sizeof(req), format, header, qty, i);
		n = henry8x_send((IFDEVICE *) dev, (unsigned char *) req, n);
		size = henry8x_recv((IFDEVICE *) dev, (unsigned char **) &pack, &err);
		if (size <= n)
			break;

		q = tokenizer(']', pack, tk, qty + 2);
		for (j = 1 ; j < q ; j++)
		{
			user = json_object_new(1);
			nro_cartao = json_array_new(1);
			cartao = json_object_new(1);
			json_object_add(user, "cracha", json_string_new(tk[j]));
			json_object_add(cartao, "nro", json_string_new(tk[j]));
			json_object_add(cartao, "tecnologia", json_string_new("Digital"));
			json_array_add(nro_cartao, cartao);
			json_object_add(user, "nro_cartao", nro_cartao);
			json_array_add(users, user);
		}

		if_free(pack);
	}

	n = getUserBioQty((IFDEVICE *) dev, users);

	henry8x_close((IFDEVICE *) dev);

	if (n < json_array_length(users))
	{
		verboseFATAL(&(dev->log), "Falha ao tentar obter quantidade de biometrias dos usuarios.\n");
		json_value_free(users);
		users = NULL;
	}

	return(users);
}
// ////////////////////////////////////////////////////////////////////////
// ////////////////////////////////////////////////////////////////////////
int henry8x_online_send_message(IFDEVICE *dev, int seq, int cod_evento, char *msg1, char *msg2, int timeout)
{
	char header[] = "REON+00";
	char req[sizeof(header) + PATH_LEN];
	int n;

	n = snprintf(req, sizeof(req), "%02d+%s+%d]%d]%s}%s]1", seq, header, cod_evento, timeout, msg1, msg2);
	n = henry8x_send(dev, (unsigned char *) req, n);

	return(n);
}
// ////////////////////////////////////////////////////////////////////////
int henry8x_online_auth(IFDEVICE *dev, int seq, JSON_VALUE *jevent, int *cod_cartao, int *cod_pessoa, int *cod_acesso)
{
	char *resp_tk[IFDEVICE_RESPONSE_LEN];
	char *txt = json_serialize(jevent);
	char *msg1, *msg2, auth;
	int timeout = json_object_get_int(dev->config, "tempo_aguardando_giro");
	int len;

	verboseDEBUG(&(dev->log), "Online_auth: -->%s<--\n", txt);
	len = tokenizer('|', txt, resp_tk, IFDEVICE_RESPONSE_LEN);
	if (len < IFDEVICE_RESPONSE_COD_CARTAO)
	{
		henry8x_online_send_message(dev, seq, 30, "Falha", "Comunicacao", 2);
		goto henry8x_online_auth_end;
	}

	*cod_cartao = atoi(resp_tk[IFDEVICE_RESPONSE_COD_CARTAO]);
	*cod_pessoa = atoi(resp_tk[IFDEVICE_RESPONSE_COD_PESSOA]);
	*cod_acesso = atoi(resp_tk[IFDEVICE_RESPONSE_ACCESS_CODE]);

	if (resp_tk[IFDEVICE_RESPONSE_AUTH][0] == 'B')
	{
		auth = 30;
	}
	else
	{
		// TODO - revisar
		//if (resp_tk[IFDEVICE_RESPONSE_TURN][0] == 'E')
		//	auth = 5;
		//else if (resp_tk[IFDEVICE_RESPONSE_TURN][0] == 'S')
		//	auth = 6;
		//else
			auth = 1;
	}

	msg1 = resp_tk[IFDEVICE_RESPONSE_MSG1];
	msg2 = resp_tk[IFDEVICE_RESPONSE_MSG2];

	if (timeout < 2)
		timeout = 3;

	henry8x_online_send_message(dev, seq, auth, msg1, msg2, timeout);

henry8x_online_auth_end:
	if_free(txt);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
int henry8x_online(IFDEVICE4J *dev, IFDEVICE4J_online_callback callback, void *ctx)
{
	char *modo = json_object_get_string(dev->config, "MODO");
	char str_cod_pessoa[20], str_cod_cartao[20], str_cod_acesso[20];
	int t, seq, size, err, tk_len, delay = 0;
	int cod_pessoa, cod_cartao, cod_acesso;
	unsigned char *pack = NULL;
	JSON_VALUE *jevent = NULL;
	char datahora[PATH_LEN];
	char *head[5], *tk[10];
	char *cartao, *sentido;

	if ((modo[0] == 0) || (modo[0] != '1'))
	{
		verboseDEBUG(&(dev->log), "Online desabilitado - verifique ifponto.ini (MODO)\n");
		return(1);
	}

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(-1);

	memset(str_cod_pessoa, 0, 20);
	memset(str_cod_cartao, 0, 20);
	memset(str_cod_acesso, 0, 20);

	verboseDEBUG(&(dev->log), "Online Start\n");
	for (t = 6 ; t > 0 ; t--, jevent = NULL, pack = NULL)
	{
		size = henry8x_recv((IFDEVICE *) dev, (unsigned char **) &pack, &err);
		if (size < 0)
		{
			delay = -1;
			verboseDEBUG(&(dev->log), "Online falha comunicacao: %d\n", size);
			break;
		}
		else if (size == 0)
		{
			// Sai do loop após 3 minutos (6 * 30) sem movimentacao
			verboseDEBUG(&(dev->log), "Online count: %d\n", t);
			continue;
		}
		else
			t = 6;
        
		tk_len = tokenizer('+', (char *) pack, head, 5);
		if (tk_len < 4)
		{
			verboseDEBUG(&(dev->log), "Online falha comunicacao: 'Pacote corrompido'\n");
			if_free(pack);
			delay = -2;
			break;
		}

		seq = atoi(head[0]);
        
		tk_len = tokenizer(']', head[3], tk, 8);
		if (tk_len < 6)
		{
			if_free(pack);
			delay = -3;
			break;
		}
        
		if (tk[3][0] == '1')
			sentido = "E";
		else if (tk[3][0] == '2')
			sentido = "S";
		else
			sentido = "I";
        
		if (head[3][0] == '0')
		{
			snprintf(datahora, PATH_LEN, "20%c%c-%c%c-%c%c %s",
				tk[2][8], tk[2][9], tk[2][3], tk[2][4], tk[2][0], tk[2][1], tk[2] + 11);
        
			for (cartao = tk[1] ; cartao[0] == '0' ; cartao++)
				;
      
			char *req_list[] = {"cartao",cartao, "sentido",sentido, "canal",tk[5], "datahora",datahora};
			jevent = callback(sizeof(req_list) / sizeof(char *), req_list, ctx, "getUserOnline");
        
			// Responder para a catraca
			henry8x_online_auth(dev, seq, jevent, &cod_cartao, &cod_pessoa, &cod_acesso);

			snprintf(str_cod_cartao, sizeof(str_cod_cartao), "%d", cod_cartao);
			snprintf(str_cod_pessoa, sizeof(str_cod_pessoa), "%d", cod_pessoa);
			snprintf(str_cod_acesso, sizeof(str_cod_acesso), "%d", cod_acesso);
		}
		else if (strcmp(head[3] ,"81") == 0)
		{
			char *req_list[] = {"sentido",sentido, "cod_cartao",str_cod_cartao, "cod_pessoa",str_cod_pessoa, "cod_acesso",str_cod_acesso};
			jevent = callback(sizeof(req_list) / sizeof(char *), req_list, ctx, "turnConfirmation");
		}
		else
		{
			verboseDEBUG(&(dev->log), "Online - codigo: %s.\n", head[3]);
		}
        
		json_value_free(jevent);
		if_free(pack);
	}
	verboseDEBUG(&(dev->log), "Online End\n");

	henry8x_close((IFDEVICE *) dev);

	return(delay);
}
// ////////////////////////////////////////////////////////////////////////
void henry8x_init(IFDEVICE4J *dev)
{
	dev->getInfo = (IFDEVICE4J_getInfo) henry8x_getInfo;
	dev->getTime = (IFDEVICE4J_getTime) henry8x_getTime;
	dev->setTime = henry8x_setTime;
	dev->sendUsers = henry8x_sendUsers;
	dev->getUsers = henry8x_getUsers;
	dev->sendBio = henry8x_sendBio;
	dev->getBio = henry8x_getBio;
	dev->getEvents = henry8x_getEvents;
	dev->online = henry8x_online;
	dev->free = NULL;
}
// ////////////////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////// //
// Retorna Henry Checksum (XOR a partir do byte 1)
_PUBLIC unsigned char henry8x_checksum(_IN unsigned char *data, _IN int size)
{
	unsigned char checksum;
	int i, data_len;

	data_len = size;
	checksum = ((data_len >> 8) & 0xFF) ^ (data_len & 0xFF);
	for (i = 0 ; i < size ; i++)
		checksum ^= data[i];

	return(checksum);
}
// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
_PUBLIC int henry8x_connect(IFDEVICE *dev)
{
	if (dev == NULL)
		return(-1);

	dev->sock = openTCP(dev->host, dev->port, 20);

	return(dev->sock);
}
// ///////////////////////////////////////////////////////// //
_PUBLIC int henry8x_close(IFDEVICE *dev)
{
	if (dev == NULL)
		return(-1);

	if (dev->sock > 0)
		return(if_closesocket(dev->sock));

	return(0);
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
int henry8x_read_bytes(IFDEVICE *dev, void *buf, int qtd, int timeout)
{
	int n = read_bytes(dev->sock, buf, qtd, timeout);
	char *log;

	if (n < 1)
		return(n);

	log = siin_hexlog((unsigned char *) buf, n);
	verboseDEBUG(&(dev->log), "iFractal <-- leitor   bytes: %d\n%s\n\n", n, log);
	if_free(log);
	fflush(stdout);

	return(n);
}
// ///////////////////////////////////////////////////////// //
int henry8x_send_bytes(IFDEVICE *dev, void *buf, int qtd, int timeout)
{
	char *log = siin_hexlog((unsigned char *) buf, qtd);

	verboseDEBUG(&(dev->log), "iFractal --> leitor   bytes: %d\n%s\n\n", qtd, log);
	if_free(log);
	fflush(stdout);

	return(send_bytes(dev->sock, buf, qtd, timeout));
}
// ///////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////// //
_PUBLIC int henry8x_send(_IN IFDEVICE *dev, _IN unsigned char *pack, _IN int size)
{
	unsigned char header[H8x_HEADER_LEN] = {HENRY8x_SB, 0, 0};
	unsigned char trailer[H8x_TRAILER_LEN] = {0, HENRY8x_EB};
	int data_len, len = 0, n;

	data_len = size;
	header[H8x_LLEN] = data_len & 0xFF;
	header[H8x_HLEN] = (data_len >> 8) & 0xFF;
	
	if ((len = henry8x_send_bytes(dev, header, H8x_HEADER_LEN, 1)) < H8x_HEADER_LEN)
		return(-2);

	if ((n = henry8x_send_bytes(dev, pack, size, 5)) < size)
		return(-4);

	len += n;

	trailer[H8x_CS] = henry8x_checksum(pack, size);
	if ((n = henry8x_send_bytes(dev, trailer, H8x_TRAILER_LEN, 1)) < H8x_TRAILER_LEN)
		return(-5);

	len += n;
	return(len);
}
// ///////////////////////////////////////////////////////// //
_PUBLIC int henry8x_recv(_IN IFDEVICE *dev, _OUT unsigned char **pack, _OUT int *err)
{
	unsigned char header[H8x_HEADER_LEN];
	unsigned char trailer[H8x_TRAILER_LEN];
	unsigned char checksum, *buf = NULL, *dup;
	int n, size, e;
	char *tk[10];

	*pack = NULL;

	n = henry8x_read_bytes(dev, header, H8x_HEADER_LEN, 30);
	if (n < 0)
	{
		e = n;
		goto henry8x_recv_err;
	}
	else if (n == 0)
	{
		e = 0;
		goto henry8x_recv_err;
	}

	if (n < H8x_HEADER_LEN)
	{
		e = -2;
		goto henry8x_recv_err;
	}

	if (header[H8x_SB] != HENRY8x_SB)
	{
		e = -3;
		goto henry8x_recv_err;
	}

	size = header[H8x_LLEN] + 256 * header[H8x_HLEN];
	buf = if_malloc(size + 1);
	buf[size] = 0;
	n = henry8x_read_bytes(dev, buf, size, 50);
	if (n < size)
	{
		e = -4;
		goto henry8x_recv_err;
	}

	checksum = henry8x_checksum(buf, size);
	n = henry8x_read_bytes(dev, trailer, H8x_TRAILER_LEN, 10);
	if (n < H8x_TRAILER_LEN)
	{
		e = -5;
		goto henry8x_recv_err;
	}

	if (checksum != trailer[H8x_CS])
	{
		//verbose(stderr, "|%s| Falha de sincronizacao - Checksum: %02X/%02X (bug Henry ignorado)\n", 
		//	MODULE, checksum, trailer[H8x_CS]);
	}

	if (trailer[H8x_EB] != HENRY8x_EB)
	{
		e = -7;
		goto henry8x_recv_err;
	}

	dup = if_malloc(size + 1);
	memcpy(dup, buf, size + 1);
	n = tokenizer('+', (char *) dup, tk, 6);
	if (n > 4)
		e = atoi(tk[4]);
	else if (n > 2)
		e = atoi(tk[2]);
	else
		e = 17;

	if_free(dup);

	if (err != NULL)
		*err = e;

	*pack = buf;
	return(size);

henry8x_recv_err:

	if (buf != NULL)
		if_free(buf);

	if (err != NULL)
		*err = e;

	*pack = NULL;

	return(e);
}

// ///////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////// //
int henry8x_deleteAll(IFDEVICE *dev) 
{
	char header[] = "00+ED+00+C";
	char req[sizeof(header) + 20];
	unsigned char *resp = NULL;
	int n, err;

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(-1);

	snprintf(req, sizeof(req), "%s", header);
	if ((n = henry8x_send(dev, (unsigned char *) req, strlen(req))) < 0)
		goto henry8x_deleteAll_err;

	n = henry8x_recv(dev, &resp, &err);
	if ((n <= 0) || (resp == NULL))
		goto henry8x_deleteAll_err;

	if_free(resp);

	henry8x_close((IFDEVICE *) dev);
	return(1);

henry8x_deleteAll_err:
	henry8x_close((IFDEVICE *) dev);
	return(-2);

}
// ///////////////////////////////////////////////////////// //
JSON_VALUE* henry8x_getListBio(IFDEVICE *dev) 
{
	char header[] = "00+RD+00+L]1000}0";
	char req[sizeof(header) + 20];
	unsigned char *resp = NULL;
	int n, err;

	if ((henry8x_connect((IFDEVICE *) dev)) < 1)
		return(NULL);

	snprintf(req, sizeof(req), "%s", header);
	if ((n = henry8x_send(dev, (unsigned char *) req, strlen(req))) < 0)
		goto henry8x_getListBio_err;

	n = henry8x_recv(dev, &resp, &err);
	if ((n <= 0) || (resp == NULL))
		goto henry8x_getListBio_err;

	if_free(resp);

	henry8x_close((IFDEVICE *) dev);
	return(NULL);

henry8x_getListBio_err:

	henry8x_close((IFDEVICE *) dev);
	return(NULL);
}

// ///////////////////////////////////////////////////////// //



#ifdef STANDALONE

IF_GETOPT configs[] = {
	{0, 'h', IF_GETOPT_TYPE_STRING, "host", "", 0, "IP do equipamento."},
	{0, 'p', IF_GETOPT_TYPE_STRING, "port", "3000", 0, "Porta (UDP) do equipamento."},
	{0, 'l', IF_GETOPT_TYPE_BOOLEAN, "list", "", 0, "Recupera lista de indices de biometria."},
	{0, 'c', IF_GETOPT_TYPE_NUMBER, "events", "1", 0, "Recupera eventos."},
	{0, 'x', IF_GETOPT_TYPE_BOOLEAN, "delete-all", "", 0, "Exclui todas as biometrias."},
	{0, 0, 0, 0, 0, 0}
};

// ///////////////////////////////////////////////////////// //

#define DESCRIPT	"Modulo 'standalone' para configuracao/operacao de equipamentos " \
			"de controle de acesso e ponto henry8x."


// ///////////////////////////////////////////////////////////////////// //
int henry8x_run(char *host, char *port)
{
	char *fields[] = {"nro","1", "codigo","0", "verbosity","5", "modelo","henry8x", "ip",host, "porta",port, NULL,NULL};
	JSON_VALUE *jconfig = json_object_new_list(fields);
	JSON_VALUE *jresp = NULL;
	struct tm *dt;
	IFDEVICE *dev;
	int r = 0;

	dev = if_malloc(sizeof(IFDEVICE_GENERIC));
	IFPONTO_device_init(dev, jconfig);
	henry8x_init(dev);
	dev->config = jconfig;

	time_t ut = time(NULL);
	ut = henry8x_getTime(dev, 0);
	dt = localtime(&ut);
	int nsr = 1;

	verboseINFO(&(dev->log), "%02d/%02d/%d %02d:%02d:%02d\n",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min, dt->tm_sec);

	henry8x_setTime(dev, 0);
	henry8x_getInfo(dev, 0);

	if (if_getopt_isChecked(configs, "events"))
	{
		nsr = atoi(if_getopt_getValue(configs, "events"));
		if (nsr > 0)
		{
			fprintf(stdout, "Coleta 50 eventos a partir do NSR: %d\n", nsr);
			jresp = henry8x_getEvents(dev, nsr);
			fprintf(stdout, "%s\n", json_serialize(jresp));
		}
		else
			fprintf(stderr, "NSR invalido: %d\n", nsr);
	}

	if (if_getopt_isChecked(configs, "list"))
		jresp = henry8x_getListBio(dev);

	if (if_getopt_isChecked(configs, "delete-all"))
		henry8x_deleteAll(dev);
	
	return(r);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char **argv)
{
	int r = if_getopt(configs, argc, argv);

	if ((r < 0) || (!if_getopt_isChecked(configs, "host")))
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -h <IP> -c <NSR>\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\tshell$ %s -h 192.168.1.201 -c 100\n", argv[0]);
		fprintf(stderr, "\n");

		return(1);
	}

	char *host = if_getopt_getValue(configs, "host");
	char *port = if_getopt_getValue(configs, "port");
	r = 0;

	henry8x_run(host, port);

	return(r);
}

#endif
