#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <errno.h>
#include <stdarg.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef __APPLE__
#include <sys/wait.h>
#else
#include <wait.h>
#endif

#include <ifractal.h>


#define MAX_FILE_LEN	100 * 1024 * 1024

// Globais
int pid, pin[2], pout[2], perr[2];

// ////////////////////////////////////////////////////////////////
void sighandler(int s)
{
	switch (s)
	{
		case SIGSEGV:
		case SIGABRT:
			fprintf(stdout, "Propaga sinal: %d\n", s);
			kill(pid, s);
			break;

		default:
			fprintf(stdout, "Ignora sinal: %d\n", s);
			break;
	}
}
// ////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////
int verbose(FILE *fd, char *format, ...)
{
	va_list va;
	time_t now = time(NULL);
	struct tm *dt;
	int len = 0;

	dt = localtime(&now);
	len = fprintf(fd, "%02d/%02d/%d %02d:%02d:%02d;",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min,dt->tm_sec);

	va_start(va, format);
	len += vfprintf(fd, format, va);
	va_end(va);

	return(len);
}
// ////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////
int startApp(char **args, char *path)
{
	// Pipes entre o monitor (PAI)
	// e stdin/stdout/stderr da aplicacao (FILHO)
	if ((pipe(pin) < 0) || (pipe(pout) < 0) || (pipe(perr) < 0))
	{
		 verbose(stderr, "Erro PIPE\n");
		return(0);
	}

	pid = fork();
	if (pid < 0)
	{
		verbose(stderr, "Erro no Fork\n");
		return(0);
	}
	
	// Filho
	if (pid == 0)
	{
		// Posiciona o diretorio do processo
		if (chdir(path))
		{
			verbose(stderr, "Erro diretorio: %s\n", path);
			exit(1);
		} 
			
		//////////////////////////////////////////////////
		// redireciona entrada/saida/erro padrao do filho
		// p/ os pipes (pin, pout, perr) do pai
		
		// stdout
		close(pout[0]);
		dup2(pout[1], 1);
		
		// stderr
		close(perr[0]);
		dup2(perr[1], 2);
		
		// stdin
		close(pin[1]);
		//dup2(pin[0], 0);
		///////////////////////////////////////////////////
			
		// executa o processo
		execv(args[0], args);
		
		// Se a aplicacao existir e tudo der certo
		// esse codigo nunca serah executado
		verbose(stderr, "%s nao existe...\n", args[0]);
		exit(1);
	} 
	
	// PAI continua
	// Fecha a "ponta" de saida (stdout) do pipe
	close(pout[1]);
	
	// Fecha a "ponta" de saida (stderr) do pipe
	close(perr[1]);
		
	// Fecha a "ponta" de entrada (stdin) do pipe
	close(pin[0]);
	
	// PID do filho (aplicacao)
	return(pid);
}
// ////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////
int readln_timeout(int fd, char *szBuff, int max, int time)
{
	int n_select;
	fd_set fdread;
	fd_set fd_exc;
	struct timeval timeout;
	char *p;

	FD_ZERO(&fdread);
	FD_ZERO(&fd_exc);
	FD_SET(fd, &fdread);
	FD_SET(fd, &fd_exc);

	timeout.tv_sec = time;
	timeout.tv_usec = 0;

	n_select = select(fd + 1, &fdread, (fd_set *)0, &fd_exc, &timeout);

	// 0 - timeout
	// < 0 - erro
	if (n_select <= 0)
		  return(n_select);

	if (FD_ISSET(fd, &fdread))
	{
		p = szBuff;

		while ((p - szBuff) < max)
		{
			if(read(fd, p++, 1) < 1)
				return(-1);

			if (*(p - 1) == '\n')
				break;
		}
	
		*p = 0;
	}

	return(p - szBuff);
}
// ////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////
_PRIVATE void log_size_exceeded(FILE *fd)
{
	static int logged = 0;
	char buf[PATH_LEN];

	if (logged)
		return;

	snprintf(buf, PATH_LEN, "Arquivo truncado por tamanho (%d Mb)\n", MAX_FILE_LEN / (1024 * 1024));
	verbose(fd, buf);
	verbose(stderr, buf);

	logged = 1;
}
// ////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////
FILE * run(char *path, int *status)
{
	int pipes[] = {pout[0], 0, perr[0], 0, -1};
	int test, ultimo_dia = 0, i, n;
	char buf[BUFFER_LEN + 1];
	int file_len = 0;
	FILE *fd = NULL;
	struct tm *dt;
	time_t now;

	do
	{
		now = time(NULL);
		dt = localtime(&now);

		if (ultimo_dia != dt->tm_mday)
		{
			if (fd != NULL)
				fclose(fd);
			
			sprintf(buf, "%s%d%02d%02d.log", path,
				dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
				
			fd = fopen(buf, "a");
			if (fd == NULL)
			{
				verbose(stderr, "Erro ao tentar abrir %s\n", buf);
				fd = stdout;
			}
		
			ultimo_dia = dt->tm_mday;
			file_len = 0;
		}

		for (i = 0, test = 0 ; pipes[i] >= 0 ; i += 2)
		{
			n = readln_timeout(pipes[i], buf, BUFFER_LEN, 0);
			if (n > 0)
			{
				buf[n] = 0;

				if (file_len > MAX_FILE_LEN)
					log_size_exceeded(fd);
				else
					file_len += verbose(fd, "%s", buf);

				pipes[i + 1] = n;
				test++;
			}
		}

		if (test > 0)
			fflush(fd);
		else
			sleep(1);
		
		if (waitpid(pid, status, WNOHANG) == pid)
			break;
			
	} while ((pipes[1] >= 0) || (pipes[3] >= 0));

	return (fd);
}
// ////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////
_PRIVATE void main_help(char *app_name, char *descript)
{
	int va = FWVER_A, vb = FWVER_B, vc = FWVER_C, rev = FWREV;
	time_t unixtime = RELEASE_UNIXTIME;
	struct tm *dt;

	fprintf(stderr, "\n####################################");
	fprintf(stderr, "\niFractal Desenvolvimento de Software");
	fprintf(stderr, "\n####################################\n\n");

	fprintf(stderr, "Programa: %s\n", app_name);
	fprintf(stderr, "Versao: %d.%d.%d r%d", va, vb, vc, rev);

	dt = localtime(&unixtime);
	fprintf(stderr, " - %02d/%02d/%d %02d:%02d",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min);

	fprintf(stderr, "\nDescricao:\n\t%s\n\n", descript);
}
// ////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	int status, sinal, ret;
	char buf[BUFFER_LEN];
	FILE * fd;

	if (argc < 4)
	{
		main_help(argv[0], "Gerenciador de Logs.");

		fprintf(stderr,"Uso:\n\tshell$ %s <PATH_LOG> <PATH_PROC> <processo> [param1] ...\n\n", argv[0]);
		return(1);
	}

	signal(SIGHUP, sighandler);
	signal(SIGQUIT, sighandler);
	signal(SIGILL, sighandler);
	signal(SIGIOT, sighandler);		
	signal(SIGBUS, sighandler);		
	signal(SIGKILL, sighandler);		
	signal(SIGUSR1, sighandler);		
	signal(SIGUSR2, sighandler);
	signal(SIGPIPE, sighandler);
	signal(SIGCHLD, sighandler);

	do
	{
		sinal = 0;
		pid = startApp(&(argv[3]), argv[2]);
		verbose(stderr, "Inicia \"%s\" (%d)\n", argv[3], pid);
        
		// Le stdout, stderr do processo
		fd = run(argv[1], &status);
        
		if (fd == NULL)
		{
			verbose(stderr, "Encerra sem dados.\n");
			return (2);
		}
        
		// Relatorio
		verbose(stderr, "Encerra \"%s\" (%d)\n", argv[3], pid);
		verbose(fd, "Encerra \"%s\" (%d)\n", argv[3], pid);
        
		if (WIFEXITED(status))
		{
			verbose(stderr, "Normal\n");
			verbose(fd, "Normal\n");
		}
		else
		{
			verbose(stderr, "sinalizado..\n");
			verbose(fd, "sinalizado...\n");
		}
        
		ret = WEXITSTATUS(status);
		verbose(stderr, "Retorno: %d\n", ret);
		verbose(fd, "Retorno: %d\n", ret);
			
		if (WIFSIGNALED(status))
		{
			sinal = WTERMSIG(status);
			verbose(stderr, "Sinal: %d\n", sinal);
			verbose(fd, "Sinal: %d\n", sinal);
        
			switch(WTERMSIG(status))
			{
#ifndef __APPLE__
				case SIGSTKFLT: snprintf(buf, BUFFER_LEN, "SIGSTKFLT: Stack fault.\n"); break;
				case SIGPWR: snprintf(buf, BUFFER_LEN, "SIGPWR: Power failure restart (System V).\n"); break;
#endif
				case SIGHUP: snprintf(buf, BUFFER_LEN, "SIGHUP: Hangup (POSIX).\n"); break;
				case SIGINT: snprintf(buf, BUFFER_LEN, "SIGINT: Interrupt (ANSI).\n"); break;
				case SIGQUIT: snprintf(buf, BUFFER_LEN, "SIGQUIT: Quit (POSIX).\n"); break;
				case SIGILL: snprintf(buf, BUFFER_LEN, "SIGILL: Illegal instruction (ANSI).\n"); break;
				case SIGTRAP: snprintf(buf, BUFFER_LEN, "SIGTRAP: Trace trap (POSIX).\n"); break;
				case SIGABRT: snprintf(buf, BUFFER_LEN, "SIGABRT: Abort (ANSI).\n"); break;
				case SIGBUS: snprintf(buf, BUFFER_LEN, "SIGBUS: BUS error (4.2 BSD).\n"); break;
				case SIGFPE: snprintf(buf, BUFFER_LEN, "SIGFPE: Floating-point exception (ANSI).\n"); break;
				case SIGKILL: snprintf(buf, BUFFER_LEN, "SIGKILL: Kill, unblockable (POSIX).\n"); break;
				case SIGUSR1: snprintf(buf, BUFFER_LEN, "SIGUSR: 1 User-defined signal 1 (POSIX).\n"); break;
				case SIGSEGV: snprintf(buf, BUFFER_LEN, "SIGSEGV: Segmentation violation (ANSI).\n"); break;
				case SIGUSR2: snprintf(buf, BUFFER_LEN, "SIGUSR: 2 User-defined signal 2 (POSIX).\n"); break;
				case SIGPIPE: snprintf(buf, BUFFER_LEN, "SIGPIPE: Broken pipe (POSIX).\n"); break;
				case SIGALRM: snprintf(buf, BUFFER_LEN, "SIGALRM: Alarm clock (POSIX).\n"); break;
				case SIGTERM: snprintf(buf, BUFFER_LEN, "SIGTERM: Termination (ANSI).\n"); break;
				case SIGCHLD: snprintf(buf, BUFFER_LEN, "SIGCHLD: Child status has changed (POSIX).\n"); break;
				case SIGCONT: snprintf(buf, BUFFER_LEN, "SIGCONT: Continue (POSIX).\n"); break;
				case SIGSTOP: snprintf(buf, BUFFER_LEN, "SIGSTOP: Stop, unblockable (POSIX).\n"); break;
				case SIGTSTP: snprintf(buf, BUFFER_LEN, "SIGTSTP: Keyboard stop (POSIX).\n"); break;
				case SIGTTIN: snprintf(buf, BUFFER_LEN, "SIGTTIN: Background read from tty (POSIX).\n"); break;
				case SIGTTOU: snprintf(buf, BUFFER_LEN, "SIGTTOU: Background write to tty (POSIX).\n"); break;
				case SIGURG: snprintf(buf, BUFFER_LEN, "SIGURG: Urgent condition on socket (4.2 BSD).\n"); break;
				case SIGXCPU: snprintf(buf, BUFFER_LEN, "SIGXCPU: CPU limit exceeded (4.2 BSD).\n"); break;
				case SIGXFSZ: snprintf(buf, BUFFER_LEN, "SIGXFSZ: File size limit exceeded (4.2 BSD).\n"); break;
				case SIGVTALRM: snprintf(buf, BUFFER_LEN, "SIGVTALRM: Virtual alarm clock (4.2 BSD).\n"); break;
				case SIGPROF: snprintf(buf, BUFFER_LEN, "SIGPROF: Profiling alarm clock (4.2 BSD).\n"); break;
				case SIGWINCH: snprintf(buf, BUFFER_LEN, "SIGWINCH: Window size change (4.3 BSD, Sun).\n"); break;
				case SIGIO: snprintf(buf, BUFFER_LEN, "SIGIO: I/O now possible (4.2 BSD).\n"); break;
				case SIGSYS: snprintf(buf, BUFFER_LEN, "SIGSYS: Bad system call.\n"); break;
        
				default: snprintf(buf, BUFFER_LEN, "sinal: %d - desconhecido\n", WTERMSIG(status));
			}
			verbose(fd, buf);
			verbose(stderr, buf);
		}
        
		fclose(fd);
		sleep(10);
	} while ((sinal == SIGSEGV) || (sinal == SIGABRT) || (ret == RESTART_RETURN));
	
	return(0);
}
// ////////////////////////////////////////////////////////////////

