#include <ifractal.h>
#include <math.h>

#include <gd.h>
#include <gdfontg.h>
#include <gdfonts.h>
#include <gdfontmb.h>

#ifdef WIN32
#include <io.h>
#endif


#define 	BARCODE_25i_bits 	"0000111100010010100100011010010101101000000011100101001110000001111000000010110101001011000001101100100001011011000000111001000011110100001010000100111100010010100101001010000001111100000110100100011000100101100110000011001100001101110000100010010011011001001000110100100010000111011000011000110001101010010100111000000110110000011111000000101001000111100100001011010000101000010111100001001011000100000011011001001000110001100011010110001000001100110100110010000111001000001001110100100110000110011010001101001100100001100110000111011000001000110001110011000010011100001000100101110010010010011001000010110100011010000100111000010111100000001011000101101100000011110000001010010101101001000011100100000001111001000010110001001011010100101000000110110100011010000101101000000011110100001110000100111010000111001100001001100100100111010010001000011001110001100010010110001000001101110000110010010011000010011100011000100100110010010111001000001001100101100110000011011000001000110101100011000011001100"


int start_len = 4, stop_len = 5;


// ////////////////////////////////////////////////////////////
void drawStart(gdImagePtr img, int narrow, int height, int black)
{
	int x = 0;

	//gr.fillRect(x, 0, narrow, height);
	gdImageFilledRectangle(img, x, 0, x + narrow - 1, height, black);

	x += narrow;
	// white

	x += narrow;
	//gr.fillRect(x, 0, narrow, height);
	gdImageFilledRectangle(img, x, 0, x + narrow - 1, height, black);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void drawStop(gdImagePtr img, int pos, int narrow, int height, int black)
{
	int x = pos * narrow;

	//gr.fillRect(x, 0, 3 * narrow, height);
	gdImageFilledRectangle(img, x, 0, x + 3 * narrow - 1, height, black);
	x += 3 * narrow;

	// white
	x += narrow;

	//gr.fillRect(x, 0, narrow, height);
	gdImageFilledRectangle(img, x, 0, x + narrow - 1, height, black);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void drawBarcode(gdImagePtr img, char *barcode, int narrow, int height, int black)
{
	int len = strlen(barcode);
	int bar, par, bit_index;
	int k, i, x = narrow * 4;

	if ((len % 2) != 0)
		fprintf(stderr, "Codigo de barcode invalido: '%s' (nro de digitos)\n", barcode);

	for (i = 0 ; (i + 1) < len ; i += 2)
	{
		par = (barcode[i] - '0') * 10 + (barcode[i + 1] - '0');
		if ((par > 100) || (par < 0))
		{
			fprintf(stderr, "Codigo de barcode invalido: '%s' (caracteres invalidos)\n", barcode);
			break;
		}

		bit_index = 10 * par;

		for (k = 0 ; k < 10 ; k++)
		{
			if (BARCODE_25i_bits[bit_index + k] == '0')
				bar = narrow;
			else
				bar = 3 * narrow;

			if ((k % 2) == 0)
			{
				//gr.fillRect(x, 0, bar, height);
				gdImageFilledRectangle(img, x, 0, x + bar - 1, height, black);
			}

			x += bar;
		}
	}
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int calc_width(char *barcodenumber)
{
	int len = strlen(barcodenumber) / 2;

	return(start_len + stop_len + len * 18);
}
// ////////////////////////////////////////////////////////////
gdImagePtr generateBarcode(char *barcodeNumber, int height)
{
	gdImagePtr img;
	int narrow = 1 + (height / 60);
	int black, white, width, bits;

	bits = calc_width(barcodeNumber);
	width = bits * narrow;

	img = gdImageCreate(width, height);
	white = gdImageColorAllocate(img, 255, 255, 255);
	black = gdImageColorAllocate(img, 0, 0, 0);

	gdImageFilledRectangle(img, 0, 0, width, height, white);

	drawStart(img, narrow, height, black);
	drawBarcode(img, barcodeNumber, narrow, height, black);
	drawStop(img, bits - stop_len, narrow, height, black);

	return(img);
}
// ////////////////////////////////////////////////////////////
gdImagePtr generateString(char *text, int height)
{
	int x, y, h, padding = 5;
	int black, white, width;
	gdImagePtr img;

	width = (strlen(text) * gdFontGetGiant()->w) + (padding * 2);

	h = gdFontGetGiant()->h;
	height = h + 2 * padding;

	img = gdImageCreate(width, height);
	white = gdImageColorAllocate(img, 255, 255, 255);
	black = gdImageColorAllocate(img, 0, 0, 0);

	gdImageFilledRectangle(img, 0, 0, width, height, white);

	x = padding;
	y = (height - h) / 2;
	gdImageString(img, gdFontGetGiant(), x, y, (unsigned char *) text, black);
	//gdImageString(img, gdFontGetMediumBold(), x, y, (unsigned char *) text, black);

	return(img);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
unsigned char * get_png_in(
	_IN char *text, 
	_IN int height, 
	_IN gdImagePtr (*generate)(char *, int),
	_OUT int *pnglen)
{
	unsigned char *pngfile;
	gdImagePtr img;

	img = generate(text, height);
	pngfile = gdImagePngPtr(img, pnglen);
	gdImageDestroy(img);

	return(pngfile);
}
// ////////////////////////////////////////////////////////////
unsigned char * get_barcode(_IN char *text, _IN int height, _OUT int *pnglen)
{
	return(get_png_in(text, height, generateBarcode, pnglen));
}
// ////////////////////////////////////////////////////////////
unsigned char * get_text(_IN char *text, _IN int height, _OUT int *pnglen)
{
	return(get_png_in(text, height, generateString, pnglen));
}
// ////////////////////////////////////////////////////////////


#ifdef CGI

#define DESCRIPT	"Gerador de imagem de codigo de barras."

static IF_GETOPT opts[] = {
	{0, 'b', IF_GETOPT_TYPE_STRING, "barcode", "", 0, "Sequencia de numeros do codigo de barras. (em pares)"},
	{0, 't', IF_GETOPT_TYPE_STRING, "text", "", 0, "Texto para ser convertido em imagem png."},
	{0, 'c', IF_GETOPT_TYPE_NONE, "nocgi", "", 0, "Gera PNG em stdout."},
	{0, 'h', IF_GETOPT_TYPE_NUMBER, "height", "59", 0, "Altura (em pixels) do codigo de barras."},
	{0, 0, 0, 0, 0, 0, 0}
};


// ////////////////////////////////////////////////////////////
_PRIVATE int barras_clean(_INOUT char *aux)
{
	int i, j;

	for (i = 0 ; aux[i] != 0 ; )
	{
		if ((aux[i] >= '0') && (aux[i] <= '9'))
		{
			i++;
			continue;
		}

		for (j = i ; (aux[j] = aux[j + 1]) ; j++)
			;
	}

	// Verifica se deve retirar os digitos
	if (i != 48)
		return(i);

	for (i = 11 ; aux[i] != 0 ; i += 11) 
		for (j = i ; (aux[j] = aux[j + 1]) ; j++)
			;

	return(i);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	unsigned char *pngfile;
	int r, height, pnglen;
	char *aux;

	r = if_getopt(opts, argc, argv);
	if ((r)
		|| ( (!if_getopt_isChecked(opts, "barcode")) 
		&& (!if_getopt_isChecked(opts, "text"))	)
		)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(opts);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -cb <BARRAS>\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n\tshell$ %s -cb 00191503900000537500000090152836000017821441 > aux.png\n", argv[0]);

		fprintf(stderr, "\n");

		return(r);
	}

#ifdef WIN32
	_setmode( _fileno( stdin ), _O_BINARY );
	_setmode( _fileno( stdout ), _O_BINARY );
#endif

	height = atoi(if_getopt_getValue(opts, "height"));

	aux = if_getopt_getValue(opts, "barcode");
	if (aux[0] != 0)
	{
		barras_clean(aux);
		pngfile = get_barcode(aux, height, &pnglen);
	}
	else
	{
		aux = if_getopt_getValue(opts, "text");
		pngfile = get_text(aux, height, &pnglen);
	}

	if (!if_getopt_isChecked(opts, "nocgi"))
	{
		// Envia header HTTP
		fprintf(stdout, "Content-Length: %d\r\n", pnglen);
		fprintf(stdout, "Content-Type: image/png\r\n\r\n");
	}

	fwrite(pngfile, 1, pnglen, stdout);

	free(pngfile);

	return(0);
}
// /////////////////////////////////////////////////////////////////////////////
#endif

