#include <ifractal.h>

// libcurl

//#include <curl/ws2tcpip.h>
#include <curl/curl.h>


// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC URL * url_make(_IN char *host, _IN char *port, _IN char *path)
{
	char urlstr[BUFFER_LEN];
	char *protocol = "http";

	if (strcmp(port, "443") == 0)
		protocol = "https";

	snprintf(urlstr, BUFFER_LEN, "%s://%s:%s/%s", protocol, host, port, path);

	return(url_new(urlstr));
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE void url_set_file(URL *url)
{
	int i, l, pl = strlen(url->path);

	if (pl == 0)
	{
		url->file = if_strdup("");
		return;
	}

	for (i = pl - 1 ; (url->path[i] != '/') && (i > 0) ; i--)
		;

	l = pl - ++i;

	url->file = if_malloc(l + 1);
	memcpy(url->file, url->path + i, l);
	url->file[l] = 0;
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_query(URL *url, int ref)
{
	char *p;
	int l;

	for (p = url->url + ref ; (*p != 0) ; p++)
		;

	l = (p - url->url) - ref;

	url->query = if_malloc(l + 1);
	memcpy(url->query, url->url + ref, l);
	url->query[l] = 0;

	return(ref + l + 1);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_path(URL *url, int ref)
{
	char *p;
	int l;

	for (p = url->url + ref ; (*p != '?') && (*p != 0) ; p++)
		;

	l = (p - url->url) - ref;

	url->path = if_malloc(l + 1);
	memcpy(url->path, url->url + ref, l);
	url->path[l] = 0;

	if (*p == '?')
		return(ref + l + 1);

	return(ref + l);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE void url_set_default_port(URL *url)
{
	char *protocols[] = {
		"http","80", 
		"https","443",
		"ftp","21",
		NULL
		};
	int i;

	for (i = 0 ; protocols[i] != NULL ; i += 2)
	{
		if (strcmp(protocols[i], url->protocol) != 0)
			continue;

		url->port = if_strdup(protocols[i + 1]);
		return;
	}

	fprintf(stderr, "|URL| porta default indeterminada.\n");
	url->port = if_strdup("");
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_port(URL *url, int ref)
{
	char *p = url->url + ref;
	int l;

	if (*p == '/')
	{
		url_set_default_port(url);
		return(ref + 1);
	}

	for (p++, ref++ ; (*p != '/') && (*p != 0) ; p++)
		;

	l = (p - url->url) - ref;

	if ((*p == 0) || (l < 1))
	{
		fprintf(stderr, "|URL| port indeterminado.\n");
		url->port = if_strdup("");
		return(ref + l);
	}

	url->port = if_malloc(l + 1);
	memcpy(url->port, url->url + ref, l);
	url->port[l] = 0;

	return(ref + l + 1);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_host(URL *url, int ref)
{
	char *p;
	int l;

	for (p = url->url + ref ; (*p != ':') && (*p != '/') && (*p != 0) ; p++)
		;

	l = (p - url->url) - ref;

	if ((*p == 0) || (l < 1))
	{
		fprintf(stderr, "|URL| host invalido.\n");
		url->host = if_strdup("");
		return(ref + l);
	}

	url->host = if_malloc(l + 1);
	memcpy(url->host, url->url + ref, l);
	url->host[l] = 0;

	return(ref + l);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_user(URL *url, int ref)
{
	int l, ret;
	char *p;

	for (p = url->url + ref ; (*p != '@') && (*p != 0) ; p++)
		;

	if (*p == 0)
	{
		url->user = if_strdup("");
		url->pass = if_strdup("");
		return(ref);
	}

	ret = (p - url->url) + 1;
	l = ret - ref - 1;

	url->user = if_malloc(l + 1);
	memcpy(url->user, url->url + ref, l);
	url->user[l] = 0;

	for (p = url->user ; (*p != ':') && (*p != 0) ; p++)
		;

	if (*p == ':')
	{
		*p++ = 0;
		url->pass = if_strdup(p);
	}

	return(ret);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_set_protocol(URL *url)
{
	char *p;
	int l;

	if (strlen(url->url) < 4)
		goto set_protocol_err;

	for (p = url->url ; (*p != ':') && (*p != 0) ; p++)
		;

	l = p - url->url;

	if ((*p == 0) || (*(p + 1) != '/') || (*(p + 2) != '/'))
		goto set_protocol_err;

	url->protocol = if_malloc(l + 1);
	memcpy(url->protocol, url->url, l);
	url->protocol[l] = 0;

	return(l + 3);	// "://"

set_protocol_err:
	fprintf(stderr, "|URL| protocolo invalido.\n");
	url->protocol = if_strdup("");
	return(0);
}
// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC URL * url_new(_IN char *txt)
{
	URL *url;
	int ref;

	if (txt == NULL)
		return(NULL);

	url = if_malloc(sizeof(URL));
	memset(url, 0, sizeof(URL));
	url->url = if_strdup(txt);

	ref = url_set_protocol(url);
	ref = url_set_user(url, ref);
	ref = url_set_host(url, ref);
	ref = url_set_port(url, ref);
	ref = url_set_path(url, ref);
	ref = url_set_query(url, ref);
	url_set_file(url);

	return(url);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC void url_free(_IN URL *url)
{
	if (url == NULL)
		return;

	if_free(url->url);
	if_free(url->protocol);
	if_free(url->user);
	if_free(url->pass);
	if_free(url->host);
	if_free(url->port);
	if_free(url->path);
	if_free(url->query);
	if_free(url->file);

	if_free(url);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_protocol(_IN URL *url)
{
	return(url->protocol);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_user(_IN URL *url)
{
	return(url->user);
}
// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_pass(_IN URL *url)
{
	return(url->pass);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_host(_IN URL *url)
{
	return(url->host);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_port(_IN URL *url)
{
	return(url->port);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_path(_IN URL *url)
{
	return(url->path);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_file(_IN URL *url)
{
	return(url->file);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC char * url_get_query(_IN URL *url)
{
	return(url->query);
}
// ////////////////////////////////////////////////////////////////////////// //



typedef struct
{
	size_t size;
	unsigned char **resp;
} URL_RESPONSE;

typedef struct
{
	char *post;
	size_t size;
	int pos;
} URL_POST_LIST;



// ////////////////////////////////////////////////////////////////////////// //
_CALLBACK size_t url_header_cb(char *buf, size_t size, size_t nitems, void *userdata)
{
	char *filename = (char *) userdata;
	char *str_filename = "filename=\"";
	int l = size * nitems - 1;
	char *p, *q;

	buf[l] = 0;

	p = strstr(buf, str_filename);
	if (p != NULL)
	{
		l = strlen(str_filename);
		q = filename;

		for (p += l ; (*p != '\"') && (*p != 0) ; *q++ = *p++)
			;

		*q = 0;
	}

	return (nitems * size);
}
// ////////////////////////////////////////////////////////////////////////// //
_CALLBACK size_t url_read_cb(void *ptr, size_t size, size_t nmemb, void *data)
{
	URL_POST_LIST *post = (URL_POST_LIST *) data;
	size_t len;

	len = size * nmemb;
	if (len < 1)
		return(0);

	if (len > (post->size - post->pos))
		len = post->size - post->pos;

	memcpy(ptr, post->post + post->pos, len);
	post->pos += len;

	return(len);
}
// ////////////////////////////////////////////////////////////////////////// //
_CALLBACK size_t url_write_cb(void *ptr, size_t size, size_t nmemb, void *data)
{
	URL_RESPONSE *url_resp = (URL_RESPONSE *) data;
	unsigned char *buf;
	size_t p;

	p = url_resp->size;
	if (p == 0)
	{
		url_resp->size = size * nmemb;
		buf = if_malloc(url_resp->size + 1);
	}
	else
	{
		url_resp->size += size * nmemb;
		buf = realloc(*(url_resp->resp), url_resp->size + 1);
	}

	*(url_resp->resp) = buf;

	memcpy(buf + p, ptr, size * nmemb);
	buf[url_resp->size] = 0;

	return(size * nmemb);
}
// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_make_post(_IN char **list, _OUT char **post_buf)
{
	char *post, *aux;
	int i, size, len;

	for (post = NULL, i = 0, size = 0 ; list[i] != NULL ; i++)
	{
		aux = char2http(list[i]);

		len = strlen(aux);
		if ((i % 2) == 0) aux[len] = '='; else aux[len] = '&';
		aux[len + 1] = 0;

		size += (len + 2);
		if (post == NULL)
		{
			post = aux;
			continue;
		}

		post = realloc(post, size + 2);
		len = strlen(post);
		strcpy(post + len, aux);
		if_free(aux);
	}

	*post_buf = post;

	if (post != NULL)
		len = strlen(post);
	else
		len = 0;

	return(len);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_execute_in(
	_IN URL *url, 
	_INOUT CURL *curl,
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data)
{
	URL_RESPONSE url_resp;
	CURLcode res;

	curl_easy_setopt(curl, CURLOPT_URL, url->url);
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 1);

	// Ignore PEER verification
	curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);

	//curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
	//curl_easy_setopt(curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_SSLv3);

	if (progress_cb)
	{
		curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progress_cb);
		curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, user_data);
		curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
	}

	*resp = NULL;
	url_resp.size = 0;
	url_resp.resp = resp;
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, url_write_cb);
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, &url_resp);

	curl_easy_setopt(curl, CURLOPT_HEADERDATA, url->filename);
	curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, url_header_cb);

	res = curl_easy_perform(curl);
	if (res != CURLE_OK)
	{
		if (*resp != NULL)
		{
			if_free(*resp);
			*resp = NULL;
		}

		verbose(stderr, "(%d) %s - '%s'\n", res, curl_easy_strerror(res), url->url);
		return(-3);
	}

	return(url_resp.size);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PRIVATE int url_execute_post_in(
	_IN CURL *curl,
	_IN URL *url, 
	_IN char **post_list, 
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data)
{
	URL_POST_LIST url_post;
	int len;

	if (post_list)
	{
		url_post.size = url_make_post(post_list, &(url_post.post));
		url_post.pos = 0;

		if (url_post.post != NULL)
		{
			curl_easy_setopt(curl, CURLOPT_POST, 1L);
			curl_easy_setopt(curl, CURLOPT_READFUNCTION, url_read_cb);
			curl_easy_setopt(curl, CURLOPT_READDATA, &url_post);
			curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, url_post.size);
		}
	}

	len = url_execute_in(url, curl, resp, progress_cb, user_data);

	if (post_list)
		if_free(url_post.post);

	curl_easy_cleanup(curl);

	return(len);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC int url_execute_post(
	_IN URL *url, 
	_IN char **post_list, 
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data)
{
	CURL *curl;
	int r;

	if (url == NULL)
		return(-1);

	curl = curl_easy_init();
	if (curl == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar libCurl.\n");
		return(-2);
	}

	r = url_execute_post_in(curl, url, post_list, resp, progress_cb, user_data);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////// //
int debug_callback(CURL *handle, curl_infotype type, char *data, size_t size, void *user_data)
{
	char *log, *text;
	(void)handle;
 
	switch (type)
	{
		case CURLINFO_TEXT:
			verbose(stdout, "== Info: %s", data);
		default: /* in case a new one is introduced to shock us */
			return 0;
         
		case CURLINFO_HEADER_OUT:
			text = "=> Send header";
			break;
        
		case CURLINFO_DATA_OUT:
			text = "=> Send data";
			verbose(stdout, "%s\n", text);
			return 0;
        
		case CURLINFO_SSL_DATA_OUT:
			text = "=> Send SSL data";
			verbose(stdout, "%s\n", text);
			return 0;
        
		case CURLINFO_HEADER_IN:
			text = "<= Recv header";
			break;
        
		case CURLINFO_DATA_IN:
			text = "<= Recv data";
			verbose(stdout, "%s\n", text);
			return 0;
        
		case CURLINFO_SSL_DATA_IN:
			text = "<= Recv SSL data";
			verbose(stdout, "%s\n", text);
			return 0;
	}

	log = siin_hexlog((unsigned char *) data, size);
	verbose(stdout, "%s  -   bytes: %d\n%s\n\n", text, size, log);
	if_free(log);

	return 0;
}
// ////////////////////////////////////////////////////////////////////////// //
int url_execute_post_debug(
	_IN URL *url,
	_IN char **post_list,
	_OUT unsigned char **resp,
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data)
{
	CURL *curl;
	int r;

	if (url == NULL)
		return(-1);

	curl = curl_easy_init();
	if (curl == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar libCurl.\n");
		return(-2);
	}

	curl_easy_setopt(curl, CURLOPT_DEBUGFUNCTION, debug_callback);
 	curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);

	r = url_execute_post_in(curl, url, post_list, resp, progress_cb, user_data);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC int url_execute_get(
	_IN URL *url, 
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC cb, 
	_INOUT void *user_data)
{
	return(url_execute_post(url, NULL, resp, cb, user_data));
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC int url_execute_post_auth(
	_IN URL *url, 
	_IN char **post_list, 
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data,
	_IN char *user,
	_IN char *pass)
{
	char auth[PATH_LEN];
	CURL *curl;
	int r;

	if (url == NULL) 
		return(-1);

	curl = curl_easy_init();
	if (curl == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar libCurl.\n");
		return(-2);
	}

	if ((user != NULL) || (pass != NULL))
	{
		snprintf(auth, PATH_LEN, "%s:%s", user, pass);
		curl_easy_setopt(curl, CURLOPT_HTTPAUTH, (long) CURLAUTH_ANY);
		curl_easy_setopt(curl, CURLOPT_USERPWD, auth);
	}

	r = url_execute_post_in(curl, url, post_list, resp, progress_cb, user_data);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////// //
_PUBLIC int url_execute_upload(
	_IN URL *url, 
	_IN char **post_list, 
	_IN char **file_list, 
	_OUT unsigned char **resp, 
	_CALLBACK URL_PROGRESS_FUNC progress_cb, 
	_INOUT void *user_data)
{
	struct curl_httppost *post = NULL;
	struct curl_httppost *last = NULL;
	CURL *curl;
	int i, len;

	if ((url == NULL) || (file_list == NULL))
		return(-1);

	curl = curl_easy_init();
	if (curl == NULL)
	{
		verbose(stderr, "Falha ao tentar iniciar libCurl.\n");
		return(-2);
	}

	for (i = 0 ; (file_list[i] != NULL) && (file_list[i + 1] != NULL) ; i += 2)
	{
		curl_formadd(&post, &last,
		        CURLFORM_COPYNAME, file_list[i],
                	CURLFORM_FILE, file_list[i + 1],
		        CURLFORM_END);
	}

	if (post_list != NULL)
	{ 
		for (i = 0 ; (post_list[i] != NULL) && (post_list[i + 1] != NULL) ; i += 2)
		{
			curl_formadd(&post, &last,
			        CURLFORM_COPYNAME, post_list[i],
			        CURLFORM_COPYCONTENTS, post_list[i + 1],
			        CURLFORM_END);
		}
	}

	struct curl_slist *headers = NULL;
	headers = curl_slist_append(headers, "Expect:");
	curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);

	curl_easy_setopt(curl, CURLOPT_HTTPPOST, post);
	len = url_execute_in(url, curl, resp, progress_cb, user_data);

	curl_formfree(post);
	curl_easy_cleanup(curl);

	return(len);
}
// ////////////////////////////////////////////////////////////////////////// //




#ifdef STANDALONE

#define DESCRIPT	"Decodifica URL."


static IF_GETOPT opts[] = {
	{0, 'u', IF_GETOPT_TYPE_STRING, "url", "", 0, "URL a ser decodificada."},
	{0, 0, 0, 0, 0, 0, 0}
};


// ////////////////////////////////////////////////////////////
int url_progress(void *user_data, double dltotal, double dlnow, double ultotal, double ulnow)
{
	FILE *fd = (FILE *) user_data;

	fprintf(fd, "UP: %g of %g  DOWN: %g of %g\n", ulnow, ultotal, dlnow, dltotal);

	return(0);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
	char *file_list[] = {"arquivo1","url.h", "arquivo2","url.c", NULL};
	char **list = NULL;
	URL *url;
	char *txt;
	int r;

	r = if_getopt(opts, argc, argv);
	if ((r < 0) || (!if_getopt_isChecked(opts, "url")))
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(opts);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -u <URL> [POST_LIST] ...\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n\tshell$ %s -u http://ifractal.srv.br/cam360/dados.php?user=teste\n", argv[0]);
		fprintf(stderr, "\n");

		return(r);
	}

	txt = if_getopt_getValue(opts, "url");

	url = url_new(txt);
	fprintf(stdout, "protocol: '%s'\n", url_get_protocol(url));
	fprintf(stdout, "user: '%s'\n", url_get_user(url));
	fprintf(stdout, "pass: '%s'\n", url_get_pass(url));
	fprintf(stdout, "host: '%s'\n", url_get_host(url));
	fprintf(stdout, "port: '%s'\n", url_get_port(url));
	fprintf(stdout, "path: '%s'\n", url_get_path(url));
	fprintf(stdout, "query: '%s'\n", url_get_query(url));
	fprintf(stdout, "file: '%s'\n", url_get_file(url));

	if (argc > 3)
		list = &(argv[r]);

	r = url_execute_post_debug(url, list, (unsigned char **) &txt, url_progress, stderr);
	//r = url_execute_upload(url, list, file_list, (unsigned char **) &txt, url_progress, stderr);
	if (r > 0)
	{
		fprintf(stdout, "(%d) -->%s<--\n", r, txt);
		if_free(txt);
	}

	url_free(url);

	return(0);
}
// ////////////////////////////////////////////////////////////
#endif


