#include <ifractal.h>
#include <tunnel.h>

#include <glib.h>


typedef struct _TUNNEL_BASE TUNNEL_BASE;

struct _TUNNEL_BASE
{
	IF_TUNNEL_TYPE type;
	TUNNEL_STATE state;
	char ip[INET_ADDRSTRLEN];
	int sock;
	char *method;
	union
	{
		char *path;
		char *clientid;
	};
	union
	{
		char *query;
		char *token;
	};
	char *response_mime;
	size_t content_length;
	time_t timeout;
	time_t start;
	time_t unixtime;
	GString *headers;
	GString *req_res;
};

typedef struct
{
	TUNNEL_BASE base;
	JSON_VALUE *info;
	gchar *header;
} TUNNEL_CLIENT;

typedef struct
{
	TUNNEL_BASE base;
} TUNNEL_ADMIN;

typedef struct
{
	TUNNEL_BASE base;
} TUNNEL_OBSERVER;

typedef union
{
	TUNNEL_BASE base;
	TUNNEL_ADMIN server;
	TUNNEL_CLIENT client;
	TUNNEL_OBSERVER observer;
} TUNNEL;

struct OBSERVER_METHOD_TABLE
{
	char *cmd;
	void (*method)(TUNNEL *, char **, void *);
	void *data;
	char *descript;
};


GSList *tunnels = NULL;
GSList *streams = NULL;
THREAD_STATE alive = TH_ALIVE;

#define DESCRIPT 	"Servidor Tunnel"

extern IF_GETOPT configs[];


// ///////////////////////////////////////////////////////////////////// //
TUNNEL * tunnel_new(IF_TUNNEL_TYPE type, int sock, char *ip)
{
	TUNNEL *tn = if_malloc(sizeof(TUNNEL));
	tn->base.type = type;
	tn->base.sock = sock;
	tn->base.path = NULL;
	tn->base.query = NULL;
	tn->base.state = TUNNEL_STATE_FIRST;
	tn->base.start = time(NULL);
	strncpy(tn->base.ip, ip, INET_ADDRSTRLEN);
	tn->base.headers = g_string_new(NULL);
	tn->base.req_res = g_string_new(NULL);

	switch (tn->base.type)
	{
		case TUNNEL_TYPE_CLIENT:
			tn->base.response_mime = MIME_PLAIN;
			break;

		case TUNNEL_TYPE_ADMIN:
			tn->base.response_mime = MIME_OCTETSTREAM;
			break;

		case TUNNEL_TYPE_OBSERVER:
			tn->base.response_mime = MIME_PLAIN;
			break;
	}

	return(tn);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_free(TUNNEL *tn)
{
	if (tn == NULL)
		return;

	switch (tn->base.type)
	{
		case TUNNEL_TYPE_CLIENT:
		case TUNNEL_TYPE_ADMIN:
		case TUNNEL_TYPE_OBSERVER:
			break;
	}

	if (tn->base.headers != NULL)
	{
		//verbose(stderr, "Headers:\n%s\n", tn->base.headers->str);
		g_string_free(tn->base.headers, TRUE);
	}

	if (tn->base.req_res != NULL)
	{
		//verbose(stderr, "Content:\n%s\n", tn->base.req_res->str);
		g_string_free(tn->base.req_res, TRUE);
	}

	g_free(tn->base.method);
	g_free(tn->base.path);
	g_free(tn->base.query);

	if_closesocket(tn->base.sock);

	if_free(tn);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
char * tunnel_get_header(TUNNEL *tn, char *header)
{
	gchar **tk = g_strsplit(tn->base.headers->str, "\n", 30);
	gchar *val = NULL;

	for (int i = 0 ; tk[i] != NULL ; i++)
	{
		if (!g_str_has_prefix(tk[i], header))
			continue;

		val = g_strdup(tk[i] + strlen(header));
		val[strlen(val) - 1] = 0;
	}	

	g_strfreev(tk);

	return(val);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void tunnel_stream_free_pair(gpointer data, gpointer user_data)
{
	TUNNEL_STREAM_PAIR *pair = (TUNNEL_STREAM_PAIR *) data;
	tunnel_stream_pair_free(pair);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_stream_process_pair(gpointer data, gpointer user_data)
{
	TUNNEL_STREAM_PAIR *pair = (TUNNEL_STREAM_PAIR *) data;
	size_t *bytes = (size_t *) user_data;
	uint8_t buf[BUFFER_LEN];
	size_t nin, nout;

	if ((pair->last == 0) || (pair->admin_sock <= 0) || (pair->client_sock <= 0))
		return;

	do
	{
		nin = read_timeout(pair->client_sock, buf, BUFFER_LEN, 0);
		if (nin < 0)
			goto tunnel_stream_process_pair_err;

		if (nin > 0)
			nin = send_bytes(pair->admin_sock, buf, nin, 30);

		nout = read_timeout(pair->admin_sock, buf, BUFFER_LEN, 0);
		if (nout < 0)
			goto tunnel_stream_process_pair_err;

		if (nout > 0)
			nout = send_bytes(pair->client_sock, buf, nout, 30);
	}
	while ((nin + nout) > 0);

	*bytes += (nin + nout);

	if ((nin + nout) > 0)
		pair->last = time(NULL);

	return;

tunnel_stream_process_pair_err:

	pair->last = 0;
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_stream_search_pair(gpointer data, gpointer user_data)
{
	TUNNEL_STREAM_PAIR *pair = (TUNNEL_STREAM_PAIR *) data;
	void **ctx = (void **) user_data;
	TUNNEL_STREAM_PAIR **out = (TUNNEL_STREAM_PAIR **) ctx[0];
	char *admin_ip = (char *) ctx[2];
	char *client_ip = (char *) ctx[3];
	int *sock = (int *) ctx[4];

	if ((pair->last == 0) || ((pair->admin_sock > 0) && (pair->client_sock > 0)))
		return;

	if ((pair->admin_sock > 0) && (client_ip != NULL))
	{
		pair->client_sock = *sock;
		strncpy(pair->client_ip, client_ip, IP_LEN);
	}
	else if ((pair->client_sock > 0) && (admin_ip != NULL))
	{
		pair->admin_sock = *sock;
		strncpy(pair->admin_ip, admin_ip, IP_LEN);
	}
	else
		return;

	*out = pair;
}
// ///////////////////////////////////////////////////////////////////// //
gpointer tunnel_stream_thread(gpointer data)
{
	TUNNEL_STREAM *tst = (TUNNEL_STREAM *) data;
	struct sockaddr cli_addr;
	int sock, cli_len = sizeof(cli_addr);
	time_t last = time(NULL);
	char ip[INET_ADDRSTRLEN];
	char *admin_ip, *client_ip;
	size_t bytes;

	tst->state = TH_ALIVE;

	do
	{
		sock = accept_timeout(tst->ssock, 0, &cli_addr, &cli_len);
		if (sock < 0)
		{
			verbose(stderr, "accept fail.\n");
			break;
		}
		else if (sock == 0)
		{
			bytes = 0;
			g_slist_foreach((GSList *) tst->pairs, tunnel_stream_process_pair, &bytes);
			if (bytes == 0)
				sleep(1);
			else
				last = time(NULL);

			continue;
		}

		last = time(NULL);
		inet_ntop(AF_INET, &(((struct sockaddr_in *) &cli_addr)->sin_addr.s_addr), ip, sizeof(ip));

		if (strcmp(ip, tst->admin_ip) == 0)
			admin_ip = ip;
		else
			client_ip = ip;

		TUNNEL_STREAM_PAIR *pair = NULL;
		void *ctx[] = {&pair, tst, admin_ip, client_ip, &sock};
		g_slist_foreach((GSList *) tst->pairs, tunnel_stream_search_pair, ctx);

		if (pair == NULL)
		{
			pair = tunnel_stream_pair_new(admin_ip, client_ip, sock);
			tst->pairs = g_slist_append(tst->pairs, pair);
		}
	}
	while ((time(NULL) - last) < 300);

	g_slist_foreach(tst->pairs, tunnel_stream_free_pair, NULL);
	g_slist_free(tst->pairs);

	tst->state = TH_ZOMBIE;

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void tunnel_get_client_callback(gpointer data, gpointer user_data)
{
	TUNNEL *tn = (TUNNEL *) data;
	void **ctx = (void **) user_data;
	char *clientid = (char *) ctx[0];
	TUNNEL_CLIENT **cli = (TUNNEL_CLIENT **) ctx[1];
	
	if (tn == NULL)
		return;

	if (tn->base.type != TUNNEL_TYPE_CLIENT)
		return;

	if (tn->base.state != TUNNEL_STATE_WAIT_REQUEST)
		return;

	if (strcmp(tn->base.clientid, clientid) != 0)
		return;

	*cli = (TUNNEL_CLIENT *) tn;
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void tunnel_find_admin_wait_callback(gpointer data, gpointer user_data)
{
	void **ctx = (void **) user_data;
	TUNNEL **admin = (TUNNEL **) ctx[0];
	TUNNEL_STATE *state = (TUNNEL_STATE *) ctx[1];
	char *clientid = (char *) ctx[2];
	char *token = (char *) ctx[3];
	TUNNEL *tn = (TUNNEL *) data;

	if (tn->base.type != TUNNEL_TYPE_ADMIN)
		return;

	if (tn->base.state != *state)
		return;

	if (strcmp(clientid, tn->base.clientid) != 0)
		return;

	if (token != NULL)
	{
		if (tn->base.token == NULL)
			return;

		if (strcmp(token, tn->base.token) != 0)
			return;
	}

	*admin = tn;
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_first(TUNNEL *tn)
{
	char buf[1024];
	int n = readln_timeout(tn->base.sock, buf, sizeof(buf), 1);

	if (n == 0)
		return(0);
	else if (n < 0)
		goto tunnel_perform_first_err;

	char *tk[5];
	n = tokenizer(' ', buf, tk, sizeof(tk) / sizeof(char *));
	if (n < 3)
		goto tunnel_perform_first_err;

	tn->base.method = g_strdup(tk[0]);

	gchar *query = g_strstr_len(tk[1], -1, "?");
	if (query != NULL)
	{
		*query++ = 0;
		tn->base.query = g_strdup(query);
	}

	tn->base.path = g_strdup(tk[1] + 1);
	tn->base.state++;

	if (tn->base.type == TUNNEL_TYPE_ADMIN)
	{
		if ((tn->base.clientid[0] == 0) && (tn->base.query != NULL))
			tn->base.type = TUNNEL_TYPE_OBSERVER;
	}

	return(n);

tunnel_perform_first_err:

	tn->base.state = TUNNEL_STATE_END;
	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
TUNNEL_STREAM * tunnel_stream_start(char **fields, TUNNEL_CLIENT **cliout, int *err)
{
	int refport = atoi(if_getopt_getValue(configs, "PORT")) + 1;
	char *clientid = json_get_list_param(fields, "clientid");
	TUNNEL_CLIENT *cli = NULL;
	void *ctx[] = {clientid, &cli};

	g_slist_foreach(tunnels, tunnel_get_client_callback, ctx);
	if (cli == NULL)
	{
		*err = RESULT_USUARIO_NAO_ENCONTRADO;
		goto tunnel_stream_start_err;
	}

	TUNNEL_STREAM *stream = tunnel_stream_new(refport, clientid, TUNNEL_STREAM_TYPE_SHELL, fields);
	if (stream == NULL)
	{
		*err = RESULT_ERRO_INTERNO;
		goto tunnel_stream_start_err;
	}

	stream->thread = g_thread_new(clientid, tunnel_stream_thread, stream);
	streams = g_slist_append(streams, stream);

	*cliout = cli;
	*err = 0;
	return(stream);

tunnel_stream_start_err:

	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_observer_shell(TUNNEL *tn, char **fields, void *data)
{
	char *list[] = {"error","0", "result","OK", NULL,NULL}; 
	char *clientid = json_get_list_param(fields, "clientid");
	JSON_VALUE *jres = json_object_new_list(list);
	char *txt = NULL;
	int err = 0;

	if ((clientid == NULL) || (clientid[0] == 0))
	{
		json_object_add(jres, "error", json_integer_new(RESULT_FALTA_PARAMETRO));
		json_object_add(jres, "result", json_string_new(result_get_msg(RESULT_FALTA_PARAMETRO)));
		goto tunnel_perform_observer_shell_end;
	}

	TUNNEL_CLIENT *cli = NULL;
	TUNNEL_STREAM *stream = tunnel_stream_start(fields, &cli, &err);
	if (stream == NULL)
	{
		json_object_add(jres, "error", json_integer_new(err));
		json_object_add(jres, "result", json_string_new(result_get_msg(err)));
		goto tunnel_perform_observer_shell_end;
	}

	char *keys[] = {
		"type", "shell", 
		"port", stream->tunnel_port, 
		"clientid", stream->clientid, 
		"admin_ip", stream->admin_ip, 
		NULL, NULL
	};
	for (int i = 0 ; keys[i + 1] != NULL ; i += 2)
		json_object_add(jres, keys[i], json_string_new(keys[i + 1]));

	// TODO - ajustar o state e o conteudo da resposta
	cli->header = g_strdup_printf("%s%s\r\n", REMOTE_HEADER, stream->tunnel_port);

tunnel_perform_observer_shell_end:

	txt = json_serialize(jres);
	g_string_append(tn->base.req_res, txt);
	if_free(txt);
	json_value_free(jres);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_observer_list(TUNNEL *tn, char **fields, void *data)
{
	char *fs[] = {"ip", tn->base.ip, "CLIENTID", "admin", NULL, NULL};
	JSON_VALUE *info = json_object_new_list(fs);
	TUNNEL *e;
	char *txt;

	g_string_append(tn->base.req_res, "[\n");

	int len = g_slist_length(tunnels);
	for (int i = 0 ; i < len ; i++)
	{
		e = (TUNNEL *) g_slist_nth_data(tunnels, i);

		if (e == NULL)
			break;

		if (e->base.type != TUNNEL_TYPE_CLIENT)
			continue;

		if (e == tn)
			continue;

		if(e->client.info == NULL)
			continue;

		json_object_add(e->client.info, "ip", json_string_new(e->base.ip));

		txt = json_serialize(e->client.info);
		g_string_append_printf(tn->base.req_res, "%s,\n", txt);
		if_free(txt);
	}

	txt = json_serialize(info);
	g_string_append_printf(tn->base.req_res, "%s]\n", txt);
	if_free(txt);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_observer_txtlist_callback(gpointer data, gpointer user_data)
{
	TUNNEL *tn = (TUNNEL *) data;
	TUNNEL *current = (TUNNEL *) user_data;

	if (tn->base.type != TUNNEL_TYPE_CLIENT)
		return;

	if (tn == current)
		return;

	g_string_append_printf(current->base.req_res, "%50s - %20s\n", 
		tn->base.clientid, 
		tn->base.ip);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_observer_txtlist(TUNNEL *tn, char **fields, void *data)
{
	g_string_append(tn->base.req_res, "\nLista Clientes:\n\n");

	g_slist_foreach(tunnels, tunnel_perform_observer_txtlist_callback, tn);

	g_string_append(tn->base.req_res, "\n");
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_observer_help(TUNNEL *tn, char **fields, void *data)
{
	struct OBSERVER_METHOD_TABLE *meths = (struct OBSERVER_METHOD_TABLE *) data;

	g_string_append(tn->base.req_res, "\n####################################");
	g_string_append(tn->base.req_res, "\niFractal Desenvolvimento de Software");
	g_string_append(tn->base.req_res, "\n####################################\n\n");

	g_string_append(tn->base.req_res, "Programa: Tunnel Server\n");
	g_string_append_printf(tn->base.req_res, "Versao: %d.%d.%d r%d", FWVER_A,FWVER_B,FWVER_C,FWREV);

	time_t compiled_time = RELEASE_UNIXTIME;
	struct tm *dt = localtime(&compiled_time);
	g_string_append_printf(tn->base.req_res, " - %02d/%02d/%d %02d:%02d\n\n",
		dt->tm_mday, dt->tm_mon + 1, dt->tm_year + 1900,
		dt->tm_hour, dt->tm_min);

	g_string_append_printf(tn->base.req_res, "Metodos:\n");
	for (int j = 0 ; meths[j].cmd != NULL ; j++)
		g_string_append_printf(tn->base.req_res, "\tcmd = %-10s  -  %s\n", meths[j].cmd, meths[j].descript);

	g_string_append_printf(tn->base.req_res, "\nURL Query:\n");
	for (int j = 0 ; fields[j + 1] != NULL ; j += 2)
		g_string_append_printf(tn->base.req_res, "\t'%s' - '%s'\n", fields[j], fields[j + 1]);

	g_string_append(tn->base.req_res, "\n");
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_observer(TUNNEL *tn)
{
	int max_fields = 20;
	char *fields[max_fields + 2];
	char *cmd = NULL;
	int i, k = 0;
	struct OBSERVER_METHOD_TABLE meths[] = {
		{"help", tunnel_perform_observer_help, meths, "Lista os metodos disponiveis."},
		{"txtlist", tunnel_perform_observer_txtlist, NULL, "Lista clientes. (TXT)"},
		{"jsonlist", tunnel_perform_observer_list, NULL, "Lista clientes. (JSON)"},
		{"shell", tunnel_perform_observer_shell, NULL, "Abre um porta para telnet. (Remote Shell)"},
		{NULL, NULL, NULL, NULL}
	};

	fields[k++] = tn->base.query;
	fields[k] = "";
	fields[k + 1] = NULL;
	fields[k + 2] = NULL;

	for (i = 0 ; tn->base.query[i] != 0 ; i++)
	{
		if ((tn->base.query[i] != '&') && (tn->base.query[i] != '='))
			continue;

		tn->base.query[i] = 0;
		fields[k++] = query_decode(tn->base.query + i + 1);
		fields[k] = NULL; 
		fields[k + 1] = NULL; 

		if (k >= max_fields)
			break;
	}

	tn->base.response_mime = MIME_PLAIN;
	tn->base.type = TUNNEL_TYPE_CLIENT;
	g_string_free(tn->base.req_res, TRUE);
	tn->base.req_res = NULL;
	tn->base.req_res = g_string_new("");

	cmd = json_get_list_param(fields, "cmd");
	if ((cmd == NULL) || (cmd[0] == 0))
	{
		tunnel_perform_observer_help(tn, fields, meths);
		return(0);
	}

	for (int j = 0 ; meths[j].cmd != NULL ; j++)
	{
		if (strcmp(meths[j].cmd, cmd) != 0)
			continue;

		meths[j].method(tn, fields, meths[j].data);
		return(0);
	}

	tunnel_perform_observer_help(tn, fields, meths);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_header(TUNNEL *tn)
{
	char *prefix = HTTP_HEADER_LENGTH;
	int l = strlen(prefix);
	char buf[1024];
	int n = readln_timeout(tn->base.sock, buf, sizeof(buf), 1);

	if (n == 0)
		return(0);
	else if (n < 0)
		goto tunnel_perform_header_err;

	if (g_str_has_prefix(buf, prefix))
	{
		tn->base.content_length = atoi(buf + l);
	}

	prefix = TIMEOUT_HEADER;
	l = strlen(prefix);
	if (g_str_has_prefix(buf, prefix))
	{
		tn->base.timeout = atoi(buf + l);
	}

	if (strlen(buf) < 4)
	{
		tn->base.state++;
	}

	g_string_append(tn->base.headers, buf);

	return(n);

tunnel_perform_header_err:

	tn->base.state = TUNNEL_STATE_END;
	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_request(TUNNEL *tn)
{
	gchar *buf;
	int n = 0;

	if (tn->base.content_length < 1)
	{
		if (tn->base.type == TUNNEL_TYPE_OBSERVER)
		{
			tunnel_perform_observer(tn);
		}
		else
		{
			g_string_prepend(tn->base.headers, "3\r\n");
			g_string_prepend(tn->base.headers, HTTP_HEADER_LENGTH);
			g_string_prepend(tn->base.headers, "Content-Type: text/plain\r\n");
			g_string_append(tn->base.req_res, "OK\n");
		}

		tn->base.state = TUNNEL_STATE_RESPONSE;
		return(0);
	}

	buf = g_strnfill(tn->base.content_length, 0);
	n = read_bytes(tn->base.sock, buf, tn->base.content_length, 60);
	if (n == 0)
		goto tunnel_perform_request_end;
	else if (n < tn->base.content_length)
		goto tunnel_perform_request_err;
        
	g_string_append(tn->base.req_res, buf);
        
	if (tn->base.type == TUNNEL_TYPE_CLIENT)
	{
		tn->base.response_mime = MIME_OCTETSTREAM;

		if (tn->base.token == NULL)
		{
			((TUNNEL_CLIENT *) tn)->info = json_parse_mem(tn->base.req_res->str);
			g_string_free(tn->base.req_res, TRUE);
			tn->base.req_res = NULL;
			tn->base.state = TUNNEL_STATE_WAIT_REQUEST;
		}
		else
			tn->base.state = TUNNEL_STATE_WAIT_RESPONSE;
	}
	else if (tn->base.type == TUNNEL_TYPE_ADMIN)
	{
		tn->base.response_mime = MIME_OCTETSTREAM;
		tn->base.state = TUNNEL_STATE_WAIT_REQUEST;
	}
	else
	{
		tunnel_perform_observer(tn);
		tn->base.state = TUNNEL_STATE_RESPONSE;
	}

tunnel_perform_request_end:
	g_free(buf);
	return(n);

tunnel_perform_request_err:
	g_free(buf);
	tn->base.state = TUNNEL_STATE_END;
	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_wait_request(TUNNEL *tn)
{
	if (tn->base.type != TUNNEL_TYPE_CLIENT)
		return(0);

	TUNNEL_ADMIN *admin = NULL;
	TUNNEL_STATE state = TUNNEL_STATE_WAIT_REQUEST;
	void *ctx[] = {&admin, &state, tn->base.clientid, NULL};
	g_slist_foreach(tunnels, tunnel_find_admin_wait_callback, ctx);

	if (admin == NULL)
		return(0);

	tn->base.req_res = admin->base.req_res;
	g_string_prepend(tn->base.req_res, admin->base.headers->str);

	admin->base.req_res = NULL;
	tn->base.token = g_strdup(admin->base.token);

	tn->base.start = time(NULL);
	admin->base.start = time(NULL);

	tn->base.state = TUNNEL_STATE_RESPONSE;
	admin->base.state = TUNNEL_STATE_WAIT_RESPONSE;

	tn->base.timeout = admin->base.timeout;

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_wait_response(TUNNEL *tn)
{
	if (tn->base.type != TUNNEL_TYPE_CLIENT)
		return(0);

	TUNNEL_ADMIN *admin = NULL;
	TUNNEL_STATE state = TUNNEL_STATE_WAIT_RESPONSE;
	void *ctx[] = {&admin, &state, tn->base.clientid, tn->base.token};
	g_slist_foreach(tunnels, tunnel_find_admin_wait_callback, ctx);

	if (admin == NULL)
		return(0);

	if (admin->base.req_res != NULL)
		g_string_free(admin->base.req_res, TRUE);

	admin->base.req_res = tn->base.req_res;
	tn->base.req_res = g_string_new("OK\n");

	tn->base.start = time(NULL);
	admin->base.start = time(NULL);

	tn->base.state = TUNNEL_STATE_RESPONSE;
	admin->base.state = TUNNEL_STATE_RESPONSE;

	return(1);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_response_header(TUNNEL_CLIENT *tn)
{
	size_t content_length = strlen(tn->base.req_res->str);
	char buf[PATH_LEN];
	snprintf(buf, PATH_LEN, "%s%ld\r\n%s%ld\r\n", HTTP_HEADER_LENGTH, content_length, TIMEOUT_HEADER, tn->base.timeout);
	char *header[] = {
		IF_HTTP_200,
		SERVER_HEADER,
		TUNNEL_AGENT,
		"\r\n",
		"Content-Type: ",
		tn->base.response_mime,
		"\r\n",
		TOKEN_HEADER,
		tn->base.token,
		"\r\n",
		buf,
		NULL
	};
	gsize n, len;

	for (int i = 0 ; header[i] != NULL ; i++)
	{
		if ((time(NULL) - tn->base.start) > 5)
		{
			verbose(stderr, "Timeout ao enviar resposta para: %s\n", tn->base.ip);
			return(-1);
		}

		len = strlen(header[i]);
		n = send_bytes(tn->base.sock, header[i], len, 5);
		if (n < len)
		{
			verbose(stderr, "Falha ao enviar header resposta para: %s\n", tn->base.ip);
			return(-2);
		}
	}

	if (tn->header != NULL)
		n = send_bytes(tn->base.sock, tn->header, strlen(tn->header), 1);

	n = send_bytes(tn->base.sock, "\r\n", 2, 1);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_response(TUNNEL *tn)
{
	size_t content_length = strlen(tn->base.req_res->str);
	int r = 0;
	gsize n;

	if (tn->base.type == TUNNEL_TYPE_CLIENT)
		tunnel_perform_response_header((TUNNEL_CLIENT *) tn);

	n = send_bytes(tn->base.sock, tn->base.req_res->str, content_length, 60);
	if (n < content_length)
	{
		verbose(stderr, "Falha ao enviar resposta: %s - %d/%d bytes\n", tn->base.ip, n, content_length);
	}

	tn->base.state++;

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //
int tunnel_perform_end(TUNNEL *tn)
{
	tn->base.state++;
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_perform_callback(gpointer data, gpointer user_data)
{
	TUNNEL *tn = (TUNNEL *) data;
	int n = 0;

	do
	{
		if (	(tn->base.state != TUNNEL_STATE_WAIT_REQUEST) 
			&& (tn->base.state != TUNNEL_STATE_WAIT_RESPONSE)
			&& (tn->base.state != TUNNEL_STATE_HEADER)
			&& (tn->base.state != TUNNEL_STATE_FREE)	)
		{
			switch (tn->base.type)
			{
				case TUNNEL_TYPE_CLIENT:
					verbose(stdout, "CLIENT - %lX\n", (uint64_t) tn);
					break;
				case TUNNEL_TYPE_ADMIN:
					verbose(stdout, "ADMIN - %lX\n", (uint64_t) tn);
					break;
				case TUNNEL_TYPE_OBSERVER:
					verbose(stdout, "OBSERVER - %lX\n", (uint64_t) tn);
					break;
			}
		}

		switch (tn->base.state)
		{
			case TUNNEL_STATE_FIRST:
				n = tunnel_perform_first(tn);
				verbose(stdout, "\tFIRST: %s?%s (%s)\n", tn->base.clientid, tn->base.token, tn->base.ip);
				break;
        
			case TUNNEL_STATE_HEADER:
				n = tunnel_perform_header(tn);
				//verbose(stdout, "\tHEADER: %s (%s)\n", tn->base.clientid, tn->base.ip);
				break;
        
			case TUNNEL_STATE_REQUEST:
				n = tunnel_perform_request(tn);
				verbose(stdout, "\tREQUEST: %s?%s (%s)\n", tn->base.clientid, tn->base.token, tn->base.ip);
				break;
        
			case TUNNEL_STATE_WAIT_REQUEST:
				n = tunnel_perform_wait_request(tn);
				break;
        
			case TUNNEL_STATE_WAIT_RESPONSE:
				n = tunnel_perform_wait_response(tn);
				break;
        
			case TUNNEL_STATE_RESPONSE:
				verbose(stdout, "\tRESPONSE: %s?%s (%s)\n", tn->base.clientid, tn->base.token, tn->base.ip);
				n = tunnel_perform_response(tn);
				break;
        
			case TUNNEL_STATE_END:
				verbose(stdout, "\tEND: %s?%s (%s)\n", tn->base.clientid, tn->base.token, tn->base.ip);
				n = tunnel_perform_end(tn);
				break;
        
			case TUNNEL_STATE_FREE:
				n = 0;
				break;
		}
	}
	while (n > 0);
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_stream_clean()
{
	TUNNEL_STREAM *tst = NULL;
	int len = g_slist_length(streams);

	for (int i = 0 ; i < len ; i++)
	{
		tst = (TUNNEL_STREAM *) g_slist_nth_data(streams, i);
		if (tst == NULL)
			continue;

		if (tst->state != TH_ZOMBIE)
			continue;

		streams = g_slist_remove(streams, tst);
		verbose(stdout, "Free stream type: %d  -  port: %s\n", tst->type, tst->tunnel_port);
		tunnel_stream_free(tst);
	}
}
// ///////////////////////////////////////////////////////////////////// //
void tunnel_clean()
{
	time_t now = time(NULL);
	int len = g_slist_length(tunnels);
	TUNNEL *tn;

	for (int i = 0 ; i < len ; i++)
	{
		tn = (TUNNEL *) g_slist_nth_data(tunnels, i);
		if (tn == NULL)
			continue;

		if (tn->base.state != TUNNEL_STATE_FREE)
		{
			if ((now - tn->base.start) < 120)
			{
				if (tn->base.state > TUNNEL_STATE_REQUEST)
					continue;
				else if ((now - tn->base.start) < 15)
					continue;
			}
		}

		tunnels = g_slist_remove(tunnels, tn);
		verbose(stdout, "Free tunnel type: %d  -  %s\n", tn->base.type, tn->base.clientid);
		tunnel_free(tn);
	}
}
// ///////////////////////////////////////////////////////////////////// //
int run()
{
	char *admins = if_getopt_getValue(configs, "ADMINS");
	char *port = if_getopt_getValue(configs, "PORT");
	struct sockaddr cli_addr;
	int ssock[1], sock, cli_len = sizeof(cli_addr);
	char ip[INET_ADDRSTRLEN];
	gchar **ips;
	TUNNEL *tn;

	ssock[0] = openServerTCP(port);
	if (ssock[0] < 1)
	{
		verbose(stderr, "Falha ao tentar abrir porta: %s\n", port);
		return(1);
	}

	ips = g_strsplit(admins, ",", 100);

	while (alive == TH_ALIVE)
	{
		g_main_context_iteration(NULL, FALSE);

		sock = accept_pool(ssock, sizeof(ssock) / sizeof(int), 0, &cli_addr, &cli_len);
		if (sock < 0)
		{
			verbose(stderr, "accept fail.\n");
			return(sock);
		}
		else if (sock > 0)
		{
			inet_ntop(AF_INET, &(((struct sockaddr_in *) &cli_addr)->sin_addr.s_addr), ip, sizeof(ip));
			if (tunnel_isAdmin(ip, ips))
				tn = tunnel_new(TUNNEL_TYPE_ADMIN, sock, ip);
			else
				tn = tunnel_new(TUNNEL_TYPE_CLIENT, sock, ip);

			tunnels = g_slist_prepend(tunnels, tn);
		}

		if_sleep(100);

		g_slist_foreach(tunnels, tunnel_perform_callback, NULL);

		tunnel_stream_clean();
		tunnel_clean();
	}

	g_strfreev(ips);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void sighandler(int s)
{
	if (s == SIGPIPE)
		return;

	alive = TH_DEAD;
	fprintf(stdout, "Signal: %d\n", s);
}
// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char *argv[])
{
	char *config_ini = TUNNEL_INI;
	int r = 0;

	if (argc < 2)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(configs);

		fprintf(stderr, "\nUso:\n\t$ %s <TUNNEL_INI>\n\n", argv[0]);
		fprintf(stderr, "Exemplo:\n\t$ %s -\n", argv[0]);
		fprintf(stderr, "\t$ %s %s\n\n", argv[0], TUNNEL_INI);
		return(1);
	}

	signal(SIGINT, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGQUIT, sighandler);
	signal(SIGPIPE, sighandler);
	signal(SIGHUP, sighandler);

	if (argv[1][0] != '-')
		config_ini = argv[1];

	if (if_getopt_ini(config_ini, configs) < 1)
	{
		verbose(stderr, "Falha ao tentar ler: %s\n", config_ini);
		verbose(stderr, "Gera arquivo: %s\n", config_ini);
		if_getopt_save(config_ini, configs);
		return(1);
	}

	r = run();

	return(r);
}
// ///////////////////////////////////////////////////////////////////// //


