#include <ifractal.h>
#include <services.h>


// ////////////////////////////////////////////////////////////
char service_type(IF_SERVICE_TYPE type)
{
	switch(type)
	{
		case IF_SERVICE_TYPE_STRING: return('S');
		case IF_SERVICE_TYPE_INTEGER: return('I');
		case IF_SERVICE_TYPE_NUMBER: return('N');
		case IF_SERVICE_TYPE_BOOLEAN: return('B');
		case IF_SERVICE_TYPE_OPTION: return('O');
	}

	return('S');
}
// ////////////////////////////////////////////////////////////
int service_show_parameters(
	_IN IF_SERVICE_CONTEXT *context, 
	_IN IF_SERVICE *service)
{
	_IN IF_SERVICE_PARAMETER_DEFINITION **params = service->parameters;
	IF_SERVICE_PARAMETER_DEFINITION *param;
	IF_SERVICE_PARAMETER_OPTION *options;
	int count_bytes = 0;
	int i, j;

	if (params == NULL)
		return(0);

	for (i = 0 ; params[i]->name != NULL ; i++)
	{
		param = params[i];
		count_bytes += printf("\t\t(%c) %-10s - %-15s - %s\n", 
			service_type(param->type), 
			param->name, 
			param->label, 
			param->description);

		if (param->options != NULL)
		{
			options = param->options;
			for (j = 0 ; ; j++)
			{
				if ((options[j].id == NULL) || (options[j].value == NULL))
					break;

				count_bytes += printf("\t\t\t(%s) - %s\n", 
					options[j].id, 
					options[j].value);
			}
		}
	}

	return(count_bytes);
}
// ////////////////////////////////////////////////////////////
int service_show_definitions(_IN IF_SERVICE_CONTEXT *context)
{
	IF_SERVICE *services = context->services;
	IF_SERVICE *serv;
	int count_bytes = 0;
	int i;

	if ((context == NULL) || (services == NULL))
		return(-1);

	for (i = 0 ; services[i].name != NULL ; i++)
	{
		serv = &(services[i]);
		count_bytes += fprintf(stdout, "\n\tFuncao: \"%s\" (%s)\n", serv->label, serv->name);
		count_bytes += fprintf(stdout, "\tDescricao: %s\n", serv->description);
		count_bytes += fprintf(stdout, "\tParametros:\n");

		count_bytes = service_show_parameters(context, serv);
	}

	return(count_bytes);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
JSON_VALUE * service_json_definition_options(_IN IF_SERVICE_PARAMETER_OPTION *options)
{
	char *list[] = {"id",NULL, "value",NULL, NULL,NULL};
	JSON_VALUE *arr, *obj;
	int i;

	arr = json_array_new(2);
	for (i = 0 ; options[i].id != NULL ; i++)
	{
		list[1] = options[i].id;
		list[3] = options[i].value;
		obj = json_object_new_list(list);

		json_array_add(arr, obj);
	}

	return(arr);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_json_definition_parameters(_IN IF_SERVICE_PARAMETER_DEFINITION **params)
{
	char *list[] = {
		"name",NULL,
		"label",NULL, 
		"description",NULL, 
		NULL,NULL};
	JSON_VALUE *aux, *arr, *obj;
	int i;

	arr = json_array_new(2);
	
	for (i = 0 ; params[i]->name != NULL ; i++)
	{
		list[1] = params[i]->name;
		list[3] = params[i]->label;
		list[5] = params[i]->description;
		obj = json_object_new_list(list);

		json_object_add(obj, "type", json_integer_new(params[i]->type));

		if (params[i]->style != NULL)
			 json_object_add(obj, "style", json_parse_mem(params[i]->style));

		if (params[i]->options != NULL)
		{
			aux = service_json_definition_options(params[i]->options);
			json_object_add(obj, "options", aux);
		}

		json_array_add(arr, obj);
	}

	return(arr);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_json_definition_services(_IN IF_SERVICE *servs)
{
	char *list[] = {
		"name",NULL,
		"label",NULL, 
		"description",NULL, 
		"style",NULL,
		NULL,NULL};
	JSON_VALUE *aux, *arr, *obj;
	int i;

	arr = json_array_new(2);
	
	for (i = 0 ; servs[i].name != NULL ; i++)
	{
		list[1] = servs[i].name;
		list[3] = servs[i].label;
		list[5] = servs[i].description;

		if (servs[i].style != NULL)
			list[7] = servs[i].style;
		else
			list[6] = NULL;

		obj = json_object_new_list(list);

		if (servs[i].parameters != NULL)
		{
			aux = service_json_definition_parameters(servs[i].parameters);
			json_object_add(obj, "parameters", aux);
		}

		if (servs[i].services != NULL)
		{
			aux = service_json_definition_services(servs[i].services);
			json_object_add(obj, "services", aux);
		}

		json_array_add(arr, obj);
	}

	return(arr);
}
// ////////////////////////////////////////////////////////////
char * service_json_definition(_IN IF_SERVICE_CONTEXT *context)
{
	JSON_VALUE *aux, *json;
	char *ret;

	if ((context == NULL) || (context->services == NULL))
		return(if_strdup("[]"));

	json = json_object_new(2);

	aux = json_string_new(context->name);
	json_object_add(json, "name", aux);

	aux = service_json_definition_services(context->services);
	json_object_add(json, "services", aux);

	ret = json_serialize(json);
	json_value_free(json);

	return(ret);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
IF_SERVICE_PARAMETER * service_get_parameter_list(_IN IF_SERVICE *service)
{
	IF_SERVICE_PARAMETER *params;
	int i, size, len = 0;

	if (service->parameters != NULL)
		for (len = 0 ; service->parameters[len]->name != NULL ; len++)
			;

	size = (len + 1) * sizeof(IF_SERVICE_PARAMETER);
	params = if_malloc(size);
	memset(params, 0, size);

	for (i = 0 ; i < len ; i++)
	{
		params[i].name = service->parameters[i]->name;
		params[i].type = service->parameters[i]->type;
		params[i].value = if_strdup(service->parameters[i]->default_value);
	}

	return(params);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * service_get_parameter_value(_IN IF_SERVICE_PARAMETER *params, _IN char *name)
{
	int i;

	if (params == NULL)
		return("");

	for (i = 0 ; params[i].name != NULL ; i++)
		if (strcmp(params[i].name, name) == 0)
			return(params[i].value);

	return("");
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
void service_parameters_free(_IN IF_SERVICE_PARAMETER *params)
{
	int i;

	if (params == NULL)
		return;

	for (i = 0 ; params[i].name != NULL ; i++)
		if_free(params[i].value);

	if_free(params);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
IF_SERVICE * service_find_func(IF_SERVICE *services, char *name)
{
	int i, len;

	if (services == NULL)
		return(NULL);

	for (i = 0 ; services[i].func != NULL ; i++)
	{
		len = strlen(services[i].name);
		if (strncmp(services[i].name, name, len) != 0)
			continue;

		if (name[len] == '.')
			return(service_find_func(services[i].services, name + len + 1));

		if (name[len] != 0)
			continue;

		if (services[i].func == NULL)
			break;

		return(&(services[i]));
	}

	return(NULL);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
char * get_value_from_list(char *list[], char *name)
{
	int i;

	for (i = 0 ; list[i + 1] != NULL ; i += 2)
	{
		if (strcmp(name, list[i]) != 0)
			continue;

		return(list[i + 1]);
	}

	return(NULL);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * service_name_value_call(
	_IN char *list[],
	_INOUT IF_SERVICE_CONTEXT *context)
{
	IF_SERVICE *serv;
	IF_SERVICE_PARAMETER *params;
	char *func_name, *param_name, *param_value;
	JSON_VALUE *response;
	int i;

	if (list == NULL)
		return(NULL);

	func_name = get_value_from_list(list, "call");
	if (func_name == NULL)
	{
		func_name = get_value_from_list(list, "json-call");
		if (func_name == NULL)
			return(NULL);

		return(service_json_call(func_name, context));
	}

	serv = service_find_func(context->services, func_name);
	if (serv == NULL)
		return(NULL);

	params = service_get_parameter_list(serv);
	for (i = 0 ; params[i].name != NULL ; i++)
	{
		param_name = serv->parameters[i]->name;
		param_value = get_value_from_list(list, param_name);
		if (param_value != NULL)
		{
			if_free(params[i].value);
			params[i].value = if_strdup(param_value);
		}
	}

	response = serv->func(context, serv, params);
	service_parameters_free(params);

	return(response);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_cli_call(
	_IN int argc,
	_IN char *argv[],
	_INOUT IF_SERVICE_CONTEXT *context)
{
	char *list[argc + 2];
	int size, i;

	size = (argc + 1) * sizeof(char *);
	memset(list, 0, size);
	for (i = 1 ; i < argc ; i++)
	{
		if (strncmp("--", argv[i], 2) == 0)
			list[i - 1] = argv[i] + 2;
		else
			list[i - 1] = argv[i];
	}

	return(service_name_value_call(list, context));
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int parameter_iter(char *key, JSON_VALUE *value, void *user_data)
{
	IF_SERVICE_PARAMETER *params = (IF_SERVICE_PARAMETER *) user_data;
	int i;

	for (i = 0 ; params[i].name != NULL ; i++)
	{
		if (strcmp(params[i].name, key) != 0)
			continue;

		if_free(params[i].value);
		params[i].value = if_strdup(json_get_string(value));
	}

	return(0);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_json_call_in(
	_IN JSON_VALUE *json_call,
	_INOUT IF_SERVICE_CONTEXT *context)
{
	JSON_VALUE *aux, *json_params, *response, *json_func_name, *ret = NULL;
	IF_SERVICE *serv;
	IF_SERVICE_PARAMETER *params;
	char *func_name;

	if (json_get_type(json_call) != JSON_OBJECT_TYPE)
		return(NULL);

	aux = json_object_find(json_call, "name");
	if (aux == NULL)
		return(NULL);

	func_name = json_get_string(aux);
	json_func_name = json_string_new(func_name);
	serv = service_find_func(context->services, func_name);
	if (serv == NULL)
		return(NULL);

	params = service_get_parameter_list(serv);

	json_params = json_object_find(json_call, "parameters");
	if ((json_params != NULL) && (json_get_type(json_params) == JSON_OBJECT_TYPE))
		json_object_iter(json_params, parameter_iter, params);

	response = serv->func(context, serv, params);
	service_parameters_free(params);

	ret = json_object_new(2);

	json_object_add(ret, "name", json_func_name);

	if (response == NULL)
		aux = json_string_new("");
	else
		aux = response;

	json_object_add(ret, "response", aux);

	return(ret);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
int func_iter(JSON_VALUE *call, void *data)
{
	void **args = (void **) data;
	JSON_VALUE *aux, *response = (JSON_VALUE *) args[0];
	IF_SERVICE_CONTEXT *context = (IF_SERVICE_CONTEXT *) args[1]; 

	aux = service_json_call_in(call, context);
	if (aux != NULL)
		json_array_add(response, aux);

	return(0);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_json_call(
	_IN char *json,
	_INOUT IF_SERVICE_CONTEXT *context)
{
	JSON_VALUE *ret = NULL, *response, *calls;
	void *args[2];

	calls = json_parse_mem(json);
	if (calls == NULL)
		return(NULL);

	if (json_get_type(calls) == JSON_ARRAY_TYPE)
	{
		response = json_array_new(2);
		args[0] = response;
		args[1] = context;
		json_array_iter(calls, func_iter, args);
	}
	else
	{
		response = service_json_call_in(calls, context);
		if (response == NULL)
			goto service_call_error;
	}

	ret = response;

service_call_error:
	json_value_free(calls);

	return(ret);
}
// ////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////
int cgi_call_iter(IF_CGI_PARAM *param, void *user_data)
{
	void **data = user_data;
	char **list = data[0];
	int *pos = data[1];

	list[*pos + 0] = param->name;
	list[*pos + 1] = param->value;

	*pos += 2;

	return(0);
}
// ////////////////////////////////////////////////////////////
char ** get_list_from_cgifile(_IN IF_CGI_PARAM *params)
{
	int par_qty = cgi_params_count(params);
	void *user_data[2];
	int size, i = 0;
	char **list;

	size = (2 * par_qty + 2) * sizeof(char *);
	list = if_malloc(size);
	memset(list, 0, size);

	user_data[0] = list;
	user_data[1] = &i;
	cgi_params_iter(params, cgi_call_iter, user_data);

	return(list);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * service_cgi_call(_INOUT IF_SERVICE_CONTEXT *context)
{
	char *boundary, *content;
	IF_CGI_PARAM *params = NULL; 
	int content_length;
	char **list = NULL;
	JSON_VALUE *ret = NULL;

	fprintf(stdout, "Content-Type: text/plain\r\n\r\n");

	content_length = cgi_get_content(stdin, &content);
	boundary = cgi_get_boundary();
	if (boundary[0] != 0)
	{
		params = cgi_get_params(content, boundary, content_length);
		list = get_list_from_cgifile(params);
	}
	else
	{
		if (content[0] == 0)
		{
			if_free(content);
			content = if_strdup(cgi_get_queryString());
		}

		list = tokenizer_params(content);
	}
	ret = service_name_value_call(list, context);
	if_free(list);
	if_free(content);

	if (params != NULL)
		cgi_params_free(params);

	return(ret);
}
// ////////////////////////////////////////////////////////////

