#include <ifractal.h>
#include <server.h>
#include <main.h>

// Globais
extern IF_VERBOSITY verbosity;
extern IF_GETOPT configs[];
extern char *home_path;


// ////////////////////////////////////////////////////////////////////////
void * th_event(IF_THREAD_LIST *list, IF_THREAD_EVENT *ev)
{
	switch (ev->event)
	{
		case IF_EVENT_IDLE: if_sleep(1); break;
		case IF_EVENT_END: if_sleep(1); break;
		default: break;
	}

	return(NULL);
}
// ////////////////////////////////////////////////////////////////////////

// ///////////////////////////////////////////////////////////////////// //
int mod_config(_INOUT MODSIIN_CONTEXT *ctx)
{
	verbose(stdout, "Le configuracao: %s.\n", CONFIG_INI);
	if (if_getopt_ini(CONFIG_INI, configs) < 1)
	{
		verbose(stderr, "Falha ao tentar ler: %s\n", CONFIG_INI);
		return(11);
	}

	ctx->webport = if_getopt_getValue(configs, "PORT");

	verbosity = atoi(if_getopt_getValue(configs, "VERBOSITY"));
	verbose(stdout, "|%s| VERBOSITY = %d\n", MODULE, verbosity);

	if (verbosity >= IF_VERB_DEBUG)
	{
		verbose(stdout, "|%s| - %s\n", MODULE, CONFIG_INI);
		if_getopt_verbose(configs);
		verbose(stdout, "\n");
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////
_PUBLIC MODSIIN * modweb_new()
{
	MODSIIN *mod = (MODSIIN *) MALLOC(sizeof(MODSIIN));
	memset(mod, 0, sizeof(MODSIIN));

	//mod->version = siin_version();

	return(mod);
}
// ////////////////////////////////////////////////////////////////////////
_PUBLIC void modweb_free(MODSIIN *mod)
{
	if (mod == NULL)
		return;

	if_free(mod);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
int if_modweb_count(MODSIIN_CONTEXT *context)
{
	MODSIIN *mod;
	int count;

	for (count = 0, mod = context->modules ; mod != NULL ; mod = mod->next, count++)
		;

	return(count);
}
// ////////////////////////////////////////////////////////////////////////
MODSIIN * if_modweb_index(MODSIIN_CONTEXT *context, int index)
{
	MODSIIN *mod;
	int i;

	for (i = 0, mod = context->modules ; mod != NULL ; mod = mod->next, i++)
		if (i == index)
			return(mod);

	return(NULL);
}
// ////////////////////////////////////////////////////////////////////////
int if_modweb_iter(MODSIIN_CONTEXT *context, MODSIIN_ITER_FUNC func, void *user_data)
{
	MODSIIN *mod;
	int count;

	if (context == NULL)
		return(0);

	for (count = 0, mod = context->modules ; mod != NULL ; mod = mod->next, count++)
		if (func(mod, user_data))
			break;

	return(count);
}
// ////////////////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////////////////// //
int load_modules_iter(DIR_CONTEXT *dir, FILE_CONTEXT *fc, void *user_data)
{
	MODSIIN_CONTEXT *context = (MODSIIN_CONTEXT *) user_data;
	void *funcs[] = {"if_modweb_init", NULL, "if_modweb_finalize", NULL, "if_modweb_run", NULL, NULL, NULL};
	char *dest, *orig, *ext;
	int len, extlen, i;
	MODSIIN *mod;

#ifdef WIN32
	ext = ".dll";
#elif __linux__
	ext = ".so";
#elif __APPLE__
	ext = ".dylib";
#endif

	len = strlen(fc->name);
	extlen = strlen(ext);
	if ((len < extlen) || (strcmp(fc->name + len - extlen, ext) != 0))
		return(0);

	mod = modweb_new();
	mod->configs = context->configs;
	funcs[1] = &(mod->init);
	funcs[3] = &(mod->finalize);
	funcs[5] = &(mod->run);

	// Salva o nome do modulo (sem extensao)
	for(dest = mod->path, orig = fc->name ; (*orig != '.') && (*orig != 0) ; *dest = 0)
		*dest++ = *orig++;

#ifdef WIN32
	if ((mod->handle = LoadLibrary(fc->filename)) == NULL) 
	{
		verbose (stderr, "Erro (%d) ao tentar carregar (%s)\n", GetLastError(), fc->filename);
		modweb_free(mod);
		return(0);
	}
#else
	if ((mod->handle = dlopen(fc->filename, RTLD_LAZY)) == NULL)
	{
		verbose (stderr, "%s\n", dlerror());
		fflush(stderr);
		modweb_free(mod);
		return(0);
	}
#endif

	for (i = 0 ; funcs[i] != NULL ; i += 2)
	{
#ifdef WIN32
		*((void **) funcs[i + 1]) = GetProcAddress(mod->handle, (char *) funcs[i]);
#else
		*((void **) funcs[i + 1]) = dlsym(mod->handle, (char *) funcs[i]);
#endif
	}

	if ((mod->init == NULL) || (mod->finalize == NULL))
	{
		modweb_free(mod);
		return(0);
	}

	// anexa novo modulo
	mod->next = context->modules;
	context->modules = mod;
	
	verbose(stdout, "Modulo '%s' carregado com sucesso (%s).\n", mod->path, fc->filename);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int load_modules(MODSIIN_CONTEXT *context)
{
	char lib_dir[PATH_LEN];
	DIR_CONTEXT *dir;
	int n;

#ifdef WIN32
	sprintf(lib_dir, "%s%clib", home_path, PATH_SEPARATOR);
#else
	char path[PATH_LEN];
	getcwd(path, PATH_LEN);
	sprintf(lib_dir, "%s%clib", path, PATH_SEPARATOR);
#endif

	// TODO - permitir recarga em tempo de execucao	
	verbose(stdout, "Procura modulos em: %s.\n", lib_dir);
	dir = fs_dir_new(lib_dir);
	if (dir == NULL)
	{
		verbose(stderr, "Erro ao tentar ler: %s.\n", lib_dir);
		return(0);
	}

	n = fs_dir_iter(dir, load_modules_iter, context);
	
	fs_dir_free(dir);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //

// ////////////////////////////////////////////////////////////////////////
_CALLBACK int broadcast_response_cb(char *ip, int port, BROADCAST_DATA *data, size_t len, void *user_data)
{
	MODSIIN_CONTEXT *ctx = (MODSIIN_CONTEXT *) user_data;
	unsigned char resp_default[] = {0x4D, 0x4F, 0x5F, 0x49, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6D, 0x6F, 0x6E, 0x69, 0x74, 0x6F, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xA8, 0x00, 0x67, 0xFF, 0xFF, 0xFF, 0x00, 0xC0, 0xA8, 0x00, 0x01, 0x08, 0x08, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x02, 0x14, 0x14, 0x08, 0x04, 0x17, 0x00, 0x50, 0x01};
	char siin_prefix[] = "SIIN - ";
	int siin_prefix_offset = strlen(siin_prefix);
	int webport = atoi(ctx->webport);
	BROADCAST_DATA bd;
	struct sockaddr_in remote_addr;
	int sock, n;

	memcpy(&bd, resp_default, sizeof(bd));
	bd.foscam.port[0] = (webport & 0xFF00) >> 8;
	bd.foscam.port[1] = webport & 0xFF;

	snprintf(bd.foscam.name, sizeof(bd.foscam.name), "%s", siin_prefix);
	gethostname(bd.foscam.name + siin_prefix_offset, sizeof(bd.foscam.name) - siin_prefix_offset);
	siin_mac(bd.foscam.mac, sizeof(bd.foscam.mac));

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return(-1);

	remote_addr.sin_family = AF_INET;
	remote_addr.sin_port = htons(port);
#ifdef WIN32
	remote_addr.sin_addr.s_addr = inet_addr(ip);
#else
	inet_aton(ip, &(remote_addr.sin_addr));
#endif
	memset(remote_addr.sin_zero, 0, sizeof(remote_addr.sin_zero));

	n = sendto(sock, (char *) &bd, sizeof(bd.foscam), 0, (struct sockaddr *) &remote_addr, sizeof(remote_addr));

	if_closesocket(sock);

	return(n);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
_PUBLIC MODSIIN_CONTEXT * if_modweb_context_new(IF_GETOPT *configs)
{
	MODSIIN_CONTEXT *ctx = if_malloc(sizeof(MODSIIN_CONTEXT));

	memset(ctx, 0, sizeof(MODSIIN_CONTEXT));
	ctx->configs = configs;

	mod_config(ctx);
	ctx->th_list = if_thread_list_new(th_event);

	return(ctx);
}
// ////////////////////////////////////////////////////////////////////////
int init_modules(MODSIIN *mod, void *user_data)
{
	MODSIIN_CONTEXT *ctx = (MODSIIN_CONTEXT *) user_data;

	verbose(stdout, "Inicializa '%s'.\n", mod->path);
	if (mod->init(mod))
	{
		verbose(stderr, "Erro ao tentar inicializar '%s'.\n", mod->path);
		return(0);
	}

	// TODO - cast poiter-to-integer
	if (mod->run != NULL)
		if_thread_add(ctx->th_list, mod->run, (intptr_t) mod, NULL);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
int if_modweb_context_init(MODSIIN_CONTEXT *context)
{
	char *port;

	if_help_header(MODULE, "iFractal SIIN");

	if (context->webport == NULL)
		return(1);

	context->serversock = openServerTCP(context->webport);
	if (context->serversock < 0)
	{
		verbose(stderr, "*** Port %s nao disponivel. ***\n", context->webport);
		return(2);
	}

	if (load_modules(context) < 1)
		verbose(stderr, "Nenhum modulo encontrado.\n");
	else
		if_modweb_iter(context, init_modules, context);

#ifdef WIN32
	if_closesocket(context->serversock);
	context->serversock = openServerTCP(context->webport);
	if (context->serversock < 0)
		return(2);
#endif

	port = if_getopt_getValue(configs, "BROADCAST_PORT");
	context->broadcast = broadcast_new(port, broadcast_response_cb, context);

	verbose(stdout, "Ouvindo  -  TCP: %s, UDP: %s\n", context->webport, port);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
int finalize_modules(MODSIIN *mod, void *user_data)
{
	verbose(stdout, "Finaliza '%s'.\n", mod->path);
	mod->finalize(mod);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////
int if_modweb_context_finalize(MODSIIN_CONTEXT *context)
{
	if_closesocket(context->serversock);

	if_modweb_iter(context, finalize_modules, NULL);
	if_thread_list_finalize(context->th_list);

	//if_thread_list_wait(context->th_list);
	FREE(context->th_list);

	broadcast_free(context->broadcast);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
int check_restart(time_t now)
{
	static time_t start = 0;
	static int last_min = -1;
	int restart, h, m;
	struct tm *dt;

	if (start == 0)
		start = now;

	restart = atoi(if_getopt_getValue(configs, "RESTART"));
	if (restart > 23)
		return(0);

	dt = localtime(&now);
	if ((dt->tm_hour == restart) && ((now - start) > (1 * 3601)))
	{
		start = 0;
		return(1);
	}

	if ((last_min != dt->tm_min) && ((dt->tm_min % 5) == 0))
	{
		h = restart - dt->tm_hour - 1;
		m = 60 - dt->tm_min;
		if (h < 0)
			h += 24;

		verbose(stdout, "Restart em %02d:%02d (%02d:00)\n", h, m, restart);
		last_min = dt->tm_min;
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ////////////////////////////////////////////////////////////////////////
int modweb_main(MODSIIN_CONTEXT *ctx, THREAD_STATE (*idle)(MODSIIN_CONTEXT *, void *), void *user_data)
{
	int r;

	if ((r = if_modweb_context_init(ctx)))
		return(r);

	while (idle(ctx, user_data) == TH_ALIVE)
	{
		broadcast_events_pending(ctx->broadcast);
		if_sleep(500);
	}

	ctx->alive = TH_ZOMBIE;
	if_modweb_context_finalize(ctx);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// //////////////////////////////////////////////////////////////////
int modweb_finalize_subprocs_procs_iter(char *name, int pid, int ppid, int qty_threads, void *user_data)
{
	char *except[] = {"catch_proc",  "siin", "siin.exe", NULL};
	char *file_name = (char *) user_data;
	int i;

	for (i = 0 ; except[i] != NULL ; i++)
		if (strcmp(except[i], name) == 0)
			return(0);

	if (strcmp(file_name, name) != 0)
		return(0);

	verbose(stderr, "Mata %s (%d)\n", name, pid);

#ifdef WIN32
	HANDLE h;
	if ((h = OpenProcess(PROCESS_TERMINATE, FALSE, pid)) != NULL)
	{
		TerminateProcess(h, 0);
		CloseHandle(h);
	}
#else
	kill(pid, SIGKILL);
#endif
	
	return(0);
}
// //////////////////////////////////////////////////////////////////
int modweb_finalize_subprocs_bin_iter(DIR_CONTEXT *dir, FILE_CONTEXT *file, void *user_data)
{
	if (strcmp(file->name, PROC_NAME) == 0)
		return(0);

	fs_proc_iter(modweb_finalize_subprocs_procs_iter, file->name);

	return(0);
}
// //////////////////////////////////////////////////////////////////
int modweb_finalize_subprocs()
{
	DIR_CONTEXT *dir;
	char bin_dir[] = "bin";

	dir = fs_dir_new(bin_dir);
	if (dir == NULL)
	{
		verbose(stderr, "Falha ao tentar abrir: '%s'\n", bin_dir);
		return(-1);
	}

	fs_dir_iter(dir, modweb_finalize_subprocs_bin_iter, NULL);

	fs_dir_free(dir);

	return(0);
}
// //////////////////////////////////////////////////////////////////


