#include <ifractal.h>

#include <proc_manager.h>

#define MODULE	"PROCESS"



// ///////////////////////////////////////////////////////////////////// //
char * process_get_param(char **params, char *key)
{
	int i;

	for (i = 0 ; params[i] != NULL ; i += 2)
		if (strcmp(params[i], key) == 0)
			return(params[i + 1]);

	return("");
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
char ** copy_params(char **params)
{
	int i, len, size;
	char **copy;

	if (params == NULL)
		return(NULL);

	for (len = 0 ; params[len] != NULL ; len++)
		;

	size = sizeof(char *) * (len + 2);
	copy = if_malloc(size);
	memset(copy, 0, size);
	for (i = 0 ; i < len ; i++)
		copy[i] = if_strdup(params[i]);

	return(copy);
}
// ///////////////////////////////////////////////////////////////////// //
void free_params(char **params)
{
	int i;

	if (params == NULL)
		return;

	for (i = 0 ; params[i] != NULL ; i++)
		if_free(params[i]);

	if_free(params);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
PROCESS_CONTEXT * process_context_new(PROCESS_CALLBACKS callbacks, IF_GETOPT *opts, void *user_data)
{
	PROCESS_CONTEXT *ctx;

	ctx = if_malloc(sizeof(PROCESS_CONTEXT));
	memset(ctx, 0, sizeof(PROCESS_CONTEXT));

	ctx->callbacks = callbacks;
	ctx->procs = vetor_new();
	ctx->user_data = user_data;
	ctx->opts = opts;

	return(ctx);
}
// ///////////////////////////////////////////////////////////////////// //
void process_context_free(PROCESS_CONTEXT *ctx)
{
	if (ctx == NULL)
		return;

	process_kill_all(ctx);
	if_sleep(2000);

	vetor_free(ctx->procs);
	if_free(ctx);	
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void process_set_auto_clean(PROCESS *proc)
{
	if (proc == NULL)
		return;

	proc->auto_clean = 1;
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
PROCESS * process_add(PROCESS_CONTEXT *ctx, char **params, char **args, char *path, char **env)
{
	PROCESS *proc;

	proc = if_malloc(sizeof(PROCESS));
	memset(proc, 0, sizeof(PROCESS));

	proc->state = TH_DEAD;

	if (params != NULL)
		proc->params = copy_params(params);

	if (args != NULL)
		proc->args = copy_params(args);

	if (path != NULL)
		proc->path = if_strdup(path);
	else
		proc->path = if_strdup(".");

	if (env != NULL)
		proc->env = copy_params(env);

	if (ctx == NULL)
		return(proc);

	vetor_push(ctx->procs, proc);

	return(proc);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int process_remove(PROCESS_CONTEXT *ctx, PROCESS *proc)
{
	if ((ctx == NULL) || (proc == NULL))
		return(-1);

	if (proc->params != NULL)
		free_params(proc->params);

	if (proc->args != NULL)
		free_params(proc->args);

	if (proc->path != NULL)
		if_free(proc->path);

	if (proc->env != NULL)
		free_params(proc->env);

	vetor_remove(ctx->procs, proc);

	if_free(proc);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int process_update(PROCESS *proc, char **params, char **args, char *path, char **env)
{
	if (proc == NULL)
		return(-1);

	if (params != NULL)
	{
		free_params(proc->params);
		proc->params = copy_params(params);
	}

	if (args != NULL)
	{
		free_params(proc->args);
		proc->args = copy_params(args);
	}

	if (path != NULL)
	{
		if_free(proc->path);
		proc->path = if_strdup(path);
	}

	if (env != NULL)
	{
		free_params(proc->env);
		proc->env = copy_params(env);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int process_iter_select(VETOR *vet, int index, void *data, void *user_data)
{
	void **data_array = (void **) user_data;
	void *user_data_orig = data_array[0];
	PROCESS_SELECT_CALLBACK cb = (PROCESS_SELECT_CALLBACK) data_array[1];
	char **params = (char **) data_array[2];
	PROCESS *proc = (PROCESS *) data;
	char *key, *value;
	int i;

	if (cb == NULL)
		return(1);

	if (params == NULL)
	{
		cb(proc, NULL, user_data_orig);
		return(0);
	}

	for (i = 0 ; params[i] != NULL ; i += 2)
	{
		key = params[i];
		value = params[i + 1];
		if (strcmp(value, process_get_param(proc->params, key)) != 0)
			return(0);
	}

	cb(proc, params, user_data_orig);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_select_where_params(PROCESS_CONTEXT *ctx, PROCESS_SELECT_CALLBACK cb, char **params, void *user_data)
{
	void *data_array[3];

	data_array[0] = user_data;
	data_array[1] = cb;
	data_array[2] = params;
	vetor_iter(ctx->procs, process_iter_select, data_array);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
#ifdef WIN32
HANDLE process_start(PROCESS *proc)
{
	PROCESS_INFORMATION pi;

	if (proc == NULL)
		return(NULL);

	if ((proc->state != TH_DEAD) || (proc->state == TH_SUSPENDED))
		return(proc->pid);

	proc->pid = start_proc(
		proc->args, 
		NULL,	// enviroment 
		proc->path,
		&pi,
		NULL,
		NULL,
		NULL,
		NULL);

	if (proc->pid == NULL)
	{
		verbose(stderr, "|%s| Erro ao tentar iniciar processo.\n", MODULE);
		return(0);
	}
#else
int process_start(PROCESS *proc)
{
	if (proc == NULL)
		return(0);

	if ((proc->pid > 1) || (proc->state == TH_SUSPENDED))
		return(proc->pid);

	proc->pid = start_proc(proc->args, proc->env, proc->path, proc->pin, proc->pout, proc->perr);
	if (proc->pid < 0)
	{
		verbose(stderr, "|%s| Erro ao tentar iniciar processo. (fork: %d)\n", MODULE, proc->pid);
		return(-2);
	}
#endif

	proc->state = TH_ALIVE;
	return(proc->pid);
}
// ///////////////////////////////////////////////////////////////////// //
int process_kill(PROCESS *proc)
{
	int status = 0;

	if (proc == NULL)
		return(-1);

#ifdef WIN32
	if (proc->pid == NULL)
		return(-2);

	TerminateProcess(proc->pid, 0);
#else
	if (proc->pid < 2)
		return(0);

	kill(proc->pid, SIGKILL);
	waitpid(proc->pid, &status, 0);
#endif

	proc->state = TH_DEAD;
	proc->pid = 0;

	return(status);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int process_iter_start(VETOR *vet, int index, void *data, void *user_data)
{
	PROCESS_CONTEXT *ctx = (PROCESS_CONTEXT *) user_data;
	PROCESS *proc = (PROCESS *) data;

#ifdef WIN32
	if (proc->pid != NULL)
#else
	if (proc->pid > 1)
#endif
		return(0);

	if (process_start(proc) > 0)
		ctx->callbacks.start(proc, ctx->user_data);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_start_all(PROCESS_CONTEXT *ctx)
{
	if (ctx == NULL)
		return(-1);

	return(vetor_iter(ctx->procs, process_iter_start, ctx));
}
// ///////////////////////////////////////////////////////////////////// //
int process_iter_kill(VETOR *vet, int index, void *data, void *user_data)
{
	PROCESS *proc = (PROCESS *) data;

	process_kill(proc);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_kill_all(PROCESS_CONTEXT *ctx)
{
	if (ctx == NULL)
		return(-1);

	return(vetor_iter(ctx->procs, process_iter_kill, ctx));
}
// ///////////////////////////////////////////////////////////////////// //
int process_restart_all(PROCESS_CONTEXT *ctx)
{
	if (ctx == NULL)
		return(-1);

	process_kill_all(ctx);
	process_start_all(ctx);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int process_iter_writeln(VETOR *vet, int index, void *data, void *user_data)
{
	PROCESS *proc = (PROCESS *) data;
	char *line = (char *) user_data;

	process_writeln(proc, line);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_writeln_all(PROCESS_CONTEXT *ctx, char *line)
{
	if ((ctx == NULL) || (line == NULL))
		return(-1);

	vetor_iter(ctx->procs, process_iter_writeln, line);

	return(0);	
}
// ///////////////////////////////////////////////////////////////////// //
int process_writeln(PROCESS *proc, char *line)
{
	int len = strlen(line);

	if ((proc == NULL) || (line == NULL))
		return(-1);

	return(send_bytes(proc->pin[1], line, len, 30));
}
// ///////////////////////////////////////////////////////////////////// //



// ///////////////////////////////////////////////////////////////////// //
int process_iter_wait(VETOR *vet, int index, void *data, void *user_data)
{
	PROCESS *proc = (PROCESS *) data;

	if (proc->state != TH_ZOMBIE)
		return(0);

	if ((time(NULL) - proc->last_log) < 3)
		return(0);

#ifdef WIN32
	proc->state = TH_DEAD;
#else

	PROCESS_CONTEXT *ctx = (PROCESS_CONTEXT *) user_data;
	int status;

	if (proc->pid < 2)
		return(0);

	if (waitpid(proc->pid, &status, WNOHANG) > 0)
	{
		ctx->callbacks.end(proc, ctx->user_data);
		proc->pid = -2;
		close(proc->pin[1]);
		close(proc->pout[0]);
		close(proc->perr[0]);
		proc->state = TH_DEAD;
	}
#endif

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_iter_run(VETOR *vet, int index, void *data, void *user_data)
{
	PROCESS *proc = (PROCESS *) data;

#ifdef WIN32
	DWORD exit_code;

	if (proc->pid == NULL)
		return(0);

	if (!GetExitCodeProcess(proc->pid, &exit_code))
	{
		verbose(stderr, "Falha ao tentar recuperar 'ExitCode'\n");
		proc->state = TH_ZOMBIE;
	}
	else if (exit_code == STILL_ACTIVE)
		proc->last_log = time(NULL);
	else
		proc->state = TH_ZOMBIE;

#else	
	void **params = (void **) user_data;
	PROCESS_CONTEXT *ctx = (PROCESS_CONTEXT *) params[0];
	int *bytes = (int *) params[1];
	char *buf;
	int len, lenerr;

	if (proc->pid < 2)
		return(0);

	buf = if_malloc(BUFFER_LEN);

	len = readln_timeout(proc->pout[0], buf, BUFFER_LEN, 0);
	if (len > 0)
	{
		proc->last_log = time(NULL);
		ctx->callbacks.log(proc, buf, ctx->user_data);
		*bytes += len;
	}

	lenerr = readln_timeout(proc->perr[0], buf, BUFFER_LEN, 0);
	if (lenerr > 0)
	{
		proc->last_log = time(NULL);
		ctx->callbacks.logerr(proc, buf, ctx->user_data);
		*bytes += lenerr;
	}

	if ((len < 0) || (lenerr < 0))
		proc->state = TH_ZOMBIE;

	if_free(buf);
#endif

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int process_clean(PROCESS_CONTEXT *ctx)
{
	PROCESS *proc;
	int n = 0, i, test;

	do
	{
		for (i = 0, test = 0 ; i < ctx->procs->length ; i++)
		{
			proc = (PROCESS *) ctx->procs->vetor[i];

			if (!proc->auto_clean)
				continue;
        
			if (proc->pid >= 0)
				continue;
        
			if (proc->state != TH_DEAD)
				continue;
      
			if (ctx->callbacks.clean != NULL)
				ctx->callbacks.clean(proc, ctx->user_data);

			process_remove(ctx, proc);
			test++;
			n++;
			break;
		}
	} while (test);

	return(n);
}
// ///////////////////////////////////////////////////////////////////// //
int process_run(PROCESS_CONTEXT *ctx)
{
	THREAD_STATE state;
	void *params[2];
	int bytes;

	do
	{
		bytes = 0;
		params[0] = ctx;
		params[1] = &bytes;
		vetor_iter(ctx->procs, process_iter_run, params);
		if (bytes < 1)
			if_sleep(50);

		vetor_iter(ctx->procs, process_iter_wait, ctx);
		state = ctx->callbacks.idle(ctx);

		process_clean(ctx);
	} while (state == TH_ALIVE);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


#ifdef DEBUG_PROC

// ///////////////////////////////////////////////////////////////////// //
int start_callback(PROCESS *proc, void *user_data)
{
	fprintf(stdout, "start\n");
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int end_callback(PROCESS *proc, void *user_data)
{
	fprintf(stdout, "end\n");
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int clean_callback(PROCESS *proc, void *user_data)
{
	fprintf(stdout, "clean\n");
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
THREAD_STATE idle_callback(PROCESS_CONTEXT *ctx)
{
	static time_t start = 0;

	if (start == 0)
		start = time(NULL);

	if ((time(NULL) - start) > 15)
		return(TH_ZOMBIE);

	return(TH_ALIVE);
}
// ///////////////////////////////////////////////////////////////////// //
int log_callback(PROCESS *proc, char *line, void *user_data)
{
	fprintf(stdout, "LOG: %s", line);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
int logerr_callback(PROCESS *proc, char *line, void *user_data)
{
	fprintf(stdout, "ERR: %s", line);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //


PROCESS_CALLBACKS cbs = {start_callback, end_callback, idle_callback, log_callback, logerr_callback, clean_callback};

// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char **argv)
{
	char *params[] = {"a","1", "b","2", "c","3", NULL,NULL};
#ifdef WIN32
	char *args[] = {"\\WINDOWS\\system32\\ipconfig.exe", NULL};
#else
	char *args[] = {"/bin/ls", NULL};
#endif
	PROCESS_CONTEXT *ctx;
	PROCESS *proc;

	ctx = process_context_new(cbs, NULL, NULL);
	proc = process_add(ctx, params, args, NULL, NULL);

	process_set_auto_clean(proc);

	process_start_all(ctx);

	process_run(ctx);

	process_context_free(ctx);

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
#endif

