#include <curl/curl.h>
#include <ifractal.h>
#include <modXXXXload.h>

#define MODULE		"MODDOWNLOAD"
#define CONFIG_INI	"conf/upload.ini"

#define verbose3(ref,def,format,...)	verbose2(ref,def,stdout,MODULE,format, ##__VA_ARGS__)

#define TMP_SUFFIX	".tmp"

extern IF_GETOPT XXXXload_config[];

IF_THREAD_LIST *thread_list = NULL;
IF_VERBOSITY verbosity = IF_VERB_NONE;
int alive = 0;

typedef struct
{
	JSON_VALUE *jfiles;
	FILE *fd;
	char *url_source;
	char *inpath;
	char *prefix;
	char *suffix;
	char *userpass;
	int freq;
} DOWNLOAD_CONTEXT;


// //////////////////////////////////////////////////////////////////////
_CALLBACK long moddownload_file_is_comming(struct curl_fileinfo *finfo, DOWNLOAD_CONTEXT *ctx, int remains)
{
	char *list[] = {"orig_name",finfo->filename, NULL,NULL};
	int suffix_len = strlen(ctx->suffix);
	int len = strlen(finfo->filename);
	char local_name[PATH_LEN];
	JSON_VALUE *obj;

	if (finfo->filetype != CURLFILETYPE_FILE)
		return(CURL_CHUNK_BGN_FUNC_OK);

	if (strcmp(finfo->filename + len - suffix_len, ctx->suffix) != 0)
		return(CURL_CHUNK_BGN_FUNC_OK);

	snprintf(local_name, PATH_LEN, "%s%c%s%s%s", ctx->inpath, PATH_SEPARATOR, ctx->prefix, finfo->filename, TMP_SUFFIX);

	ctx->fd = fopen(local_name, "w");
	if (!ctx->fd) 
		return(CURL_CHUNK_BGN_FUNC_FAIL);

	obj = json_object_new_list(list);
	json_object_add(obj, "size", json_integer_new(finfo->size));
	json_object_add(obj, "local_name", json_string_new(local_name));

	json_array_add(ctx->jfiles, obj);

	return(CURL_CHUNK_BGN_FUNC_OK);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_CALLBACK long moddownload_file_is_downloaded(DOWNLOAD_CONTEXT *ctx)
{
	if(ctx->fd) 
	{
		fclose(ctx->fd);
		ctx->fd = NULL;
	}

	return CURL_CHUNK_END_FUNC_OK;
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_CALLBACK size_t moddownload_write_it(char *buff, size_t size, size_t nmemb, DOWNLOAD_CONTEXT *ctx)
{
	size_t written = 0;

	if (ctx->fd)
		written = fwrite(buff, size, nmemb, ctx->fd);

	return written;
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
int moddownload_process(DOWNLOAD_CONTEXT *ctx)
{
	int rc = CURLE_OK;
	CURL *handle;

	rc = curl_global_init(CURL_GLOBAL_ALL);
	if(rc)
	{
		verbose3(verbosity, IF_VERB_FATAL, "CURL_GLOBAL_INIT error...\n");
		alive = 0;
		return(-1);
	}

	handle = curl_easy_init();
	if(!handle)
	{
		curl_global_cleanup();
		verbose3(verbosity, IF_VERB_FATAL, "CURLE_OUT_OF_MEMORY\n");
		alive = 0;
		return(-2);
	}

	curl_easy_setopt(handle, CURLOPT_WILDCARDMATCH, 1L);
	curl_easy_setopt(handle, CURLOPT_CHUNK_BGN_FUNCTION, moddownload_file_is_comming);
	curl_easy_setopt(handle, CURLOPT_CHUNK_END_FUNCTION, moddownload_file_is_downloaded);
	curl_easy_setopt(handle, CURLOPT_WRITEFUNCTION, moddownload_write_it);

	curl_easy_setopt(handle, CURLOPT_CHUNK_DATA, ctx);
	curl_easy_setopt(handle, CURLOPT_WRITEDATA, ctx);

	//curl_easy_setopt(handle, CURLOPT_VERBOSE, 1L);

	if (ctx->userpass != NULL)
		curl_easy_setopt(handle, CURLOPT_USERPWD, ctx->userpass);

	curl_easy_setopt(handle, CURLOPT_URL, ctx->url_source);

	rc = curl_easy_perform(handle);
	switch (rc)
	{
		case 0:
			break;

		case CURLE_UNSUPPORTED_PROTOCOL:
			verbose3(verbosity, IF_VERB_FATAL, "CURLE_UNSUPPORTED_PROTOCOL (%s)\n", ctx->url_source);
			break;

		default:
			verbose3(verbosity, IF_VERB_FATAL, "CURL erro: %d\n", rc);
	}

	curl_easy_cleanup(handle);
	curl_global_cleanup();

	return(rc);
}
// //////////////////////////////////////////////////////////////////////


// //////////////////////////////////////////////////////////////////////
_PRIVATE int moddownload_delete_iter(JSON_VALUE *json, void *user_data)
{
	DOWNLOAD_CONTEXT *ctx = (DOWNLOAD_CONTEXT *) user_data;
	CURL *curl;
	CURLcode res;
	char to[PATH_LEN];
	JSON_VALUE *aux;
	char *from;

	res = curl_global_init(CURL_GLOBAL_ALL);
	if(res)
	{
		verbose3(verbosity, IF_VERB_FATAL, "(%d) CURL_GLOBAL_INIT error...\n", res);
		alive = 0;
		return(1);
	}

	curl = curl_easy_init();
	if (curl == NULL)
	{
		curl_global_cleanup();
		verbose3(verbosity, IF_VERB_FATAL, "CURL_GLOBAL_INIT error...\n");
		alive = 0;
		return(2);
	}

	aux = json_object_find(json, "orig_name");
	from = json_get_string(aux);

	if (ctx->userpass != NULL)
		curl_easy_setopt(curl, CURLOPT_USERPWD, ctx->userpass);

	snprintf(to, PATH_LEN, "DELE %s", from);
	curl_easy_setopt(curl, CURLOPT_QUOTE, to);

	//curl_easy_setopt(curl, CURLOPT_VERBOSE, 1); 

	curl_easy_setopt(curl, CURLOPT_URL, ctx->url_source);
	res = curl_easy_perform(curl);
	curl_easy_cleanup(curl);

	if (CURLE_OK != res)
	{
		alive = 0;
		verbose3(verbosity, IF_VERB_FATAL, "(%d) Falha ao tentar deletar '%s'.\n", res, from);
		return(3);
	}

	verbose3(verbosity, IF_VERB_INFO, "%s - Deletado.\n", from);

	curl_global_cleanup();

	return(0);
}
// //////////////////////////////////////////////////////////////////////
_PRIVATE int moddownload_rename_iter(JSON_VALUE *json, void *user_data)
{
	char to[PATH_LEN];
	JSON_VALUE *aux;
	char *from;
	int pos;

	aux = json_object_find(json, "local_name");
	from = json_get_string(aux);

	strncpy(to, from, PATH_LEN);
	pos = strlen(to) - strlen(TMP_SUFFIX);
	to[pos] = 0;

	if (modXXXXload_rename(to, from))
	{
		verbose3(verbosity, IF_VERB_FATAL, "Falha ao tentar renomear arquivo. (%s)\n", from);
		alive = 0;
		return(1);
	}

	verbose3(verbosity, IF_VERB_INFO, "%s - OK\n", to);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
_PRIVATE int moddownload_config_iter(JSON_VALUE *json, void *user_data)
{
	int *count = (int *) user_data;
	DOWNLOAD_CONTEXT ctx;
	char *configs[] = {
		"inpath",".", 
		"url_source","", 
		"prefix","XXXX",
		"suffix",".xml",
		"userpass","",
		"freq","5",
		NULL,NULL
		};
	int n;

	if (!alive)
		return(0);

	json_make_list_param(json, configs);

	ctx.freq = atoi(json_get_list_param(configs, "freq"));
	if ((*count % (ctx.freq * 60)) != 0)
		return(0);

	ctx.url_source = json_get_list_param(configs, "url_source");
	ctx.inpath = json_get_list_param(configs, "inpath");
	ctx.prefix = json_get_list_param(configs, "prefix");
	ctx.suffix = json_get_list_param(configs, "suffix");
	ctx.userpass = json_get_list_param(configs, "userpass");

	ctx.jfiles = json_array_new(2);

	n = moddownload_process(&ctx);
	n = json_array_length(ctx.jfiles);
	if (n > 0)
	{
		verbose3(verbosity, IF_VERB_INFO, "Arquivos baixados: %d\n", n);
		json_array_iter(ctx.jfiles, moddownload_rename_iter, &ctx);
		json_array_iter(ctx.jfiles, moddownload_delete_iter, &ctx);
	}

	json_value_free(ctx.jfiles);

	return(0);
}
// //////////////////////////////////////////////////////////////////////



#ifdef STANDALONE
#define DESCRIPT	"Processo de download de arquivos."

// //////////////////////////////////////////////////////////////////////
int main (int argc, char **argv)
{
	THREAD_STATE state = TH_ALIVE;
	int r;
	
	r = if_getopt(XXXXload_config, argc, argv);
	if (	(r) || 
		!if_getopt_isChecked(XXXXload_config, "CONFIG")
	)
	{
		if_help_header(argv[0], DESCRIPT);
		fprintf(stderr, "Ajuda:\n");
		if_getopt_help(XXXXload_config);

		fprintf(stderr, "\nUso:\n");
		fprintf(stderr, "\tshell$ %s -c <JSON_CONFIG>\n", argv[0]);

		fprintf(stderr, "\nExemplo:\n");
		fprintf(stderr, "\tshell$ %s -c upload.json\n", argv[0]);

		fprintf(stderr, "\nExemplo arquivo:\n\n%s\n\n", DOWNLOAD_JSON_CONFIG);

		fprintf(stderr, "\n");

		return(r);
	}

	verbosity = atoi(if_getopt_getValue(XXXXload_config, "VERBOSITY"));

	alive = 1;
	modXXXXload_run(&state, 1, moddownload_config_iter);

	return(0);
}
// //////////////////////////////////////////////////////////////////////

#else

// Implementacao da Interface ///////////////////////////////////////////
_PUBLIC int if_modweb_init(IF_MODWEB *mod)
{
	thread_list = if_thread_list_new(NULL);

	verbose3(verbosity, IF_VERB_INFO, "Le configuracao: %s.\n", CONFIG_INI);
	if_getopt_ini(CONFIG_INI, XXXXload_config);

	verbosity = atoi(if_getopt_getValue(XXXXload_config, "VERBOSITY"));
	verbose(stdout, "|%s| VERBOSITY = %d\n", MODULE, verbosity);

	alive = 1;
	if_thread_add(thread_list, modXXXXload_run, 1, moddownload_config_iter);
	
	modweb_define_conf(mod, XXXXload_config, CONFIG_INI);
	modweb_define_info(mod, "Download", "Modulo para download arquivos FTP.", "upload.png");

	return(0);
}
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
_PUBLIC int if_modweb_finalize(IF_MODWEB *mod)
{
	if_thread_list_finalize(thread_list);
	if_thread_list_wait(thread_list);
	FREE(thread_list);

	return(0);
}
// //////////////////////////////////////////////////////////////////////
// FIM - Implementacao da Interface /////////////////////////////////////


#endif

