/** @file json.h Biblioteca iFractal JSON
\ingroup 

\note -----
\author Felipe Peressoni Waltrick	felipe@ifractal.com.br
\version 1.1
**/


#ifndef JSON_H
#define JSON_H

#include <ifractal_utils.h>
#include <stdio.h>


/**
 * Tipos de predefinidos para objetos JSON. 
**/
typedef enum
{
	JSON_INTEGER_TYPE, 	 
	JSON_NUMBER_TYPE,	/*!< Numeros em ponto flutuante (separador decimal '.') */
	JSON_BOOLEAN_TYPE,	/*!< [true|false|TRUE|FALSE] */
	JSON_STRING_TYPE,	/*!< Texto delimitado por " ou ' */
	JSON_ARRAY_TYPE,	/*!< Vetor */
	JSON_OBJECT_TYPE,	/*!< Hash */
	JSON_NULL_TYPE,
	JSON_DATA_TYPE,		/*!< Ponteiro para estrutura de dados nativa. (nao serializavel) */
	JSON_DATETIME_TYPE,	/*!< Unixtime */
	JSON_TYPE_LEN
} JSON_TYPE;


typedef struct _JSON_ARRAY JSON_ARRAY;
typedef struct _JSON_OBJECT JSON_OBJECT;

typedef union _JSON_VALUE JSON_VALUE;

typedef struct _JSON_PROPERTY
{
	char *key;
	JSON_VALUE *value;
} JSON_PROPERTY;

struct _JSON_OBJECT
{
	JSON_TYPE type;
	JSON_PROPERTY **properties;
	int length;
	int size;
};

struct _JSON_ARRAY
{
	JSON_TYPE type;
	JSON_VALUE **values;		// Vetor de elementos
	int length;
	int size;
};

/**
 * Estrutura JSON
**/
union _JSON_VALUE
{
	struct
	{
		JSON_TYPE type;
		union
		{
			char *value;
			time_t unixtime;
		};
	};

	JSON_ARRAY *array;
	JSON_OBJECT *object;
};


_PUBLIC JSON_VALUE * json_value_new(JSON_TYPE type, void *value);

/**
 * Instacia JSON string. 
**/
_PUBLIC JSON_VALUE * json_string_new(char *);

/**
 * Instacia JSON integer. 
**/
_PUBLIC JSON_VALUE * json_integer_new(int);

/**
 * Instacia JSON number. 
**/
_PUBLIC JSON_VALUE * json_number_new(double);

/**
 * Instacia JSON unixtime. 
**/
_PUBLIC JSON_VALUE * json_unixtime_new(time_t);

/**
 * Recupera unixtime. 
**/
_PUBLIC time_t json_get_unixtime(JSON_VALUE *);

/**
 * Libera memoria de um JSON. (obrigatorio) 
**/
_PUBLIC void json_value_free(JSON_VALUE *);

/**
 * Replica a estrutura JSON.
**/
_PUBLIC JSON_VALUE * json_clone(JSON_VALUE *);

/**
 * Instancia um vetor com um tamanho inicial. 
**/
_PUBLIC JSON_VALUE * json_array_new(int init_size);

/**
 * Adiciona elemento no final do vetor.
**/
_PUBLIC int json_array_add(JSON_VALUE *array, JSON_VALUE *elemento);
_PRIVATE void json_array_free(JSON_ARRAY *);

/**
 * Retorna o elemento contido em index. NULL caso index seja invalido.
**/
_PUBLIC JSON_VALUE * json_array_index(JSON_VALUE *, int index);

/**
 * Retorna tamanho do vetor.
**/
_PUBLIC int json_array_length(JSON_VALUE *);

/**
 * Remove elemento a partir do indice.
 * Retorna o ponteiro para o elemento.
**/
_PUBLIC JSON_VALUE * json_array_remove(JSON_VALUE *, int index);

/**
 * Instacia um objeto com um tamanho inicial.
**/
_PUBLIC JSON_VALUE * json_object_new(int init_size);

/**
 * Instacia um objeto a partir de uma lista (chave, valor).
**/
_PUBLIC JSON_VALUE * json_object_new_list(_IN char *list[]);

/**
 * Adiciona uma lista. 
**/
_PUBLIC int json_object_add_list(JSON_VALUE *object, char *list[]);

/**
 * Adiciona elemento a um objeto. 
**/
_PUBLIC int json_object_add(JSON_VALUE *object, char *key, JSON_VALUE *value);
_PRIVATE void json_object_free(JSON_OBJECT *);

/**
 * Remove elemento a partir da chave
**/
_PUBLIC int json_object_remove(JSON_VALUE *object, char *key);

/**
 * Localiza um elemento a partir da chave.
**/
_PUBLIC JSON_VALUE * json_object_find(JSON_VALUE *, char *key);

/**
 * Localiza um elemento recupera string.
**/
_PUBLIC char * json_object_get_string(JSON_VALUE *, char *key);

/**
 * Localiza um elemento recupera como int.
**/
_PUBLIC int json_object_get_int(JSON_VALUE *, char *key);

/**
 * Localiza um elemento recupera como unixtime.
**/
_PUBLIC time_t json_object_get_unixtime(JSON_VALUE *, char *key);

/**
 * Retorna quantidade de elementos. 
**/
_PUBLIC int json_object_length(JSON_VALUE *);

typedef int (*JSON_OBJECT_ITER_CALLACK)(char *key, JSON_VALUE *, void *user_data);
_PUBLIC int json_object_iter(JSON_VALUE *, JSON_OBJECT_ITER_CALLACK, void *user_data);

typedef int (*JSON_ARRAY_ITER_CALLACK)(JSON_VALUE *, void *user_data);
_PUBLIC int json_array_iter(JSON_VALUE *, JSON_ARRAY_ITER_CALLACK, void *user_data);

_PUBLIC char * json_type2txt(JSON_TYPE);
_PUBLIC JSON_TYPE json_get_type(JSON_VALUE *);
_PUBLIC int json_get_int(JSON_VALUE *);
_PUBLIC double json_get_double(JSON_VALUE *);
_PUBLIC int json_get_boolean(JSON_VALUE *);
_PUBLIC char * json_get_string(JSON_VALUE *);

_PUBLIC int json_make_list_param(_IN JSON_VALUE *obj, _INOUT char **list);
_PUBLIC char * json_get_list_param(_IN char **list, _IN char *key);


typedef struct _JSON_LEX_CONTEXT
{
	int (*next_char)(struct _JSON_LEX_CONTEXT *);
	char *mem_in;
	int mem_pos_in;
	FILE *fd_in;
} JSON_LEX_CONTEXT;

#define JSON_STACK_SIZE		128

typedef struct
{
	JSON_VALUE *key;
	JSON_VALUE *val;
	JSON_VALUE *array;
	JSON_VALUE *obj;
	JSON_VALUE *values[JSON_STACK_SIZE];
	int value_pos;
	JSON_LEX_CONTEXT context;
	void *scanner;
} JSON_STACK;


void json_push(JSON_STACK *, JSON_VALUE *);
JSON_VALUE * json_current(JSON_STACK *);
JSON_VALUE * json_pop(JSON_STACK *);

void json_show_value(JSON_VALUE *);

/**
 * Instancia um JSON a partir de um buffer. 
**/
_PUBLIC JSON_VALUE * json_parse_mem(char *buf);

/**
 * Instancia um JSON a partir de um arquivo. 
**/
_PUBLIC JSON_VALUE * json_parse_file(FILE *);

/**
 * Gera a representacao serializada (string) de um JSON. 
**/
_PUBLIC char * json_serialize(JSON_VALUE *);
_PUBLIC int json_serialize_file(JSON_VALUE *, char *filename);


_PRIVATE void json_input(char *buf, int *result, int max_size, void *extra);


#define YY_INPUT(buf,result,max_size) \
{ \
	json_input(buf, &result, max_size, yyextra); \
}

#endif
