#include <ifractal.h>


// ////////////////////////////////////////////////////////////////////////
void * if_thread_event_debug(IF_THREAD_LIST *list, IF_THREAD_EVENT *ev)
{
#ifdef DEBUG
	int n = if_slist_length(list->first);

	if (ev->event != IF_EVENT_IDLE)
		fprintf(stdout, "Threads Ativas:%d\n", n);

	switch (ev->event)
	{
		case IF_EVENT_START: fprintf(stdout, "EV: start (%d)\n", ev->id); break;
		case IF_EVENT_WAIT: fprintf(stdout, "EV: wait (%d)\n", ev->id); break;
		case IF_EVENT_FREE: fprintf(stdout, "EV: free (%d)\n", ev->id); break;
		case IF_EVENT_END: fprintf(stdout, "EV: end\n"); break;
		case IF_EVENT_IDLE: if_sleep(1); break;
		default: break;
	}
#endif

	return(NULL);
}
// ////////////////////////////////////////////////////////////////////////



// ////////////////////////////////////////////////////////////////////////
void if_thread_free(IF_THREAD_LIST *list, IF_THREAD *th)
{
	IF_THREAD_EVENT ev;
	
	// Chama evento
	ev.event = IF_EVENT_FREE;
	ev.id = th->id;
	ev.data = th->data;
	if (list->event_callback != NULL)
		list->event_callback(list, &ev);

	list->first = if_slist_remove_data(list->first, th);
	FREE(th);
}
// ////////////////////////////////////////////////////////////////////////
void if_thread_finalize(IF_THREAD_LIST *list, IF_THREAD *th)
{
	IF_THREAD_EVENT ev;

	// Chama evento
	ev.event = IF_EVENT_WAIT;
	ev.id = th->id;
	ev.data = th->data;
	if (list->event_callback != NULL)
		list->event_callback(list, &ev);

#ifdef WIN32
	if (th->thread != NULL)
	{
		WaitForSingleObject(th->thread, INFINITE);
		CloseHandle(th->thread);
	}
#else
	void *status;
	pthread_join(th->thread, &status);
#endif
	th->th_state = TH_DEAD;
}
// ////////////////////////////////////////////////////////////////////////
int if_thread_clean(IF_THREAD_LIST *list)
{
	IF_THREAD *th;
	IF_SLIST *aux;
	int r = 0;
	
	if_thread_mutex_lock(list);
	for (aux = list->first ; aux != NULL ; )
	{
		th = (IF_THREAD *) aux->data;
		if (th->th_state == TH_ZOMBIE)
		{
			if_thread_finalize(list, th);
			r++;
		}
		
		if (th->th_state == TH_DEAD)
		{
			if_thread_free(list, th);
			aux = list->first;
			r++;
			continue;
		}
		
		aux = aux->next;
	}
	if_thread_mutex_unlock(list);

	return(r);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
#ifdef WIN32
DWORD WINAPI if_thread_manager(void *data)
//unsigned _stdcall if_thread_manager(void *data)
#else
static void * if_thread_manager(void *data)
#endif
{
	IF_THREAD_LIST *list = (IF_THREAD_LIST *) data;
	IF_THREAD_EVENT ev;

	while ((list->th_state == TH_ALIVE) || (list->first != NULL))
	{
		// Procedimento de limpeza
		if_thread_clean(list);

		// "Libera" mutex /////////////////
		if_sleep(100);
		// ////////////////////////////////
		
		// Chama evento
		ev.event = IF_EVENT_IDLE;
		ev.id = 0;
		ev.data = NULL;
		if (list->event_callback == NULL)
		{
			if_sleep(2);
		}
		else
		{
			// Chama evento 'ocioso'.
			if_thread_mutex_lock(list);
			list->event_callback(list, &ev);
			if_thread_mutex_unlock(list);
		}
	}
		
	// Chama evento
	ev.event = IF_EVENT_END;
	ev.id = 0;
	ev.data = NULL;
	if (list->event_callback != NULL)
		list->event_callback(list, &ev);

#ifdef WIN32
	//_endthreadex(0);
	return(0);
#else
	return(NULL);
#endif
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
IF_THREAD_LIST * if_thread_list_new(IF_THREAD_EVENT_CALLBACK cb)
{
	IF_THREAD_LIST *list = MALLOC(sizeof(IF_THREAD_LIST));

	list->th_state = TH_ALIVE;
	list->event_callback = cb;
	list->first = NULL;

#ifdef WIN32
	list->mutex = CreateMutex(NULL, FALSE, NULL);
	list->thread = CreateThread(NULL, 0, if_thread_manager, list, 0, NULL);
	//list->thread = (HANDLE) _beginthreadex(NULL, 0, if_thread_manager, list, 0, NULL);
#else
	if (pthread_mutex_init(&(list->mutex), NULL) != 0)
		fprintf(stderr, "ifractal_threads Error: pthread_mutex_init\n");

	pthread_create(&(list->thread), NULL, if_thread_manager, list);
#endif

	return(list);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
#ifdef WIN32
DWORD WINAPI if_thread_launcher(void *data)
#else
static void * if_thread_launcher(void *data)
#endif
{
	IF_THREAD *th = (IF_THREAD *) data;

	th->th_state = TH_ALIVE;
	th->func(&(th->th_state), th->id, th->data);
	th->th_state = TH_ZOMBIE;
	
#ifdef WIN32
	return(0);
#else
	return(NULL);
#endif
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
IF_THREAD * if_thread_get_id(IF_THREAD_LIST *list, intptr_t id)
{
	IF_SLIST *aux;
	IF_THREAD *th;
	
	if (list == NULL)
		return(NULL);

	if_thread_mutex_lock(list);

	for (aux = list->first ; aux != NULL ; aux = aux->next)
	{
		th = (IF_THREAD *) aux->data;
		if (th->id == id)
			break;
	}
		
	if (aux == NULL)
		th = NULL;

	if_thread_mutex_unlock(list);

	return(th);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
IF_THREAD * if_thread_add(IF_THREAD_LIST *list, IF_THREAD_FUNC func, intptr_t id, void *data)
{
	IF_THREAD *novo;
	IF_THREAD_EVENT ev;

	// Verifica se id jah existe
	if (if_thread_get_id(list, id))
	{
		verbose(stderr, "thread ID: %d duplicado.\n", id);
		return(NULL);
	}

	novo = MALLOC(sizeof(IF_THREAD));
	memset(novo, 0, sizeof(IF_THREAD));
	novo->id = id;
	novo->th_state = TH_ALIVE;
	novo->func = func;
	novo->data = data;

	novo->th_state = TH_ALIVE;
#ifdef WIN32
	if ((novo->thread = CreateThread(NULL, 0, if_thread_launcher, novo, 0, NULL)) == NULL)
	//if ((novo->thread = (HANDLE) _beginthreadex(NULL, 0, if_thread_launcher, novo, 0, NULL)) == 0)
	{
		verbose(stderr, "Qtd. de thread excedida.\n");
		novo->th_state = TH_DEAD;
	}
#else
	int n;
	if ((n = pthread_create(&(novo->thread), NULL, if_thread_launcher, novo)) != 0)
	{
		verbose(stderr, "(%d) Erro pthread_create: Qtd. de thread excedida.\n", n);
		novo->th_state = TH_DEAD;
	}
#endif
	
	if_thread_mutex_lock(list);
	list->first = if_slist_append(list->first, novo);
	if_thread_mutex_unlock(list);

	// Chama evento
	ev.event = IF_EVENT_START;
	ev.id = id;
	ev.data = data;
	if (list->event_callback != NULL)
		list->event_callback(list, &ev);

	return(novo);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
int if_thread_mutex_lock(IF_THREAD_LIST *list)
{
	if (list == NULL)
		return(0);
	
#ifdef WIN32
	WaitForSingleObject( list->mutex, INFINITE );
#else
	if (pthread_mutex_trylock(&(list->mutex)) != 0)
	{
		//fprintf(stderr, "List: %X lock colision...\n", list);
		pthread_mutex_lock(&(list->mutex));
	}
	list->delay = time(NULL);
#endif
	return(0);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
int if_thread_mutex_unlock(IF_THREAD_LIST *list)
{
	if (list == NULL)
		return(0);

#ifdef WIN32
	ReleaseMutex(list->mutex);
#else
	int d;
	//fprintf(stderr, "List: %X (unlock)\n", list);
	d = time(NULL) - list->delay;
	if (d > 3)
		fprintf(stderr, "Mutex Delay: %d\n", d);
	pthread_mutex_unlock(&(list->mutex));
#endif
	return(0);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
void if_thread_list_finalize(IF_THREAD_LIST *list)
{
	IF_THREAD *th;
	IF_SLIST *aux;

	if_thread_mutex_lock(list);

	list->th_state = TH_ZOMBIE;
	for (aux = list->first ; aux != NULL ; aux = aux->next)
	{
		th = (IF_THREAD *) aux->data;
		if (th->th_state == TH_ALIVE)
			th->th_state = TH_ZOMBIE;
	}
	
	if_thread_mutex_unlock(list);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
void if_thread_stop(IF_THREAD_LIST *list, intptr_t id)
{
	IF_THREAD *th;
	IF_SLIST *aux;

	if_thread_mutex_lock(list);
	
	for (aux = list->first ; aux != NULL ; aux = aux->next)
	{
		th = (IF_THREAD *) aux->data;
		if (th->id == id)
		{
			if (th->th_state != TH_DEAD)
			{
#ifdef WIN32
				TerminateThread(th->thread, 0);
				verbose(stderr, "TerminateThread (%d)\n", id);
#else
				pthread_cancel(th->thread);
				verbose(stderr, "CancelThread (%d)\n", id);
#endif
			}

			if (th->th_state == TH_ALIVE)
				th->th_state = TH_ZOMBIE;
			break;
		}
	}
	
	if_thread_mutex_unlock(list);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
int if_thread_list_wait(IF_THREAD_LIST *list)
{
#ifdef WIN32
	WaitForSingleObject(list->thread, INFINITE);
#else
	// TODO - falha quando o numero maximo de threads eh ultrapassado
	void *status;
	pthread_join(list->thread, &status);
#endif

	return(0);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
int if_thread_iter(IF_THREAD_LIST *list, THREAD_ITER_FUNC iter_func, void *user_data)
{
	IF_THREAD *th;
	IF_SLIST *aux;
	int qty;

	if_thread_mutex_lock(list);
	
	for (aux = list->first, qty = 0 ; aux != NULL ; aux = aux->next, qty++)
	{
		th = (IF_THREAD *) aux->data;

		iter_func(th, user_data);
	}

	if_thread_mutex_unlock(list);

	return(qty);
}
// ////////////////////////////////////////////////////////////////////////




#ifdef DEBUG_THREAD
int main_alive;
IF_THREAD_LIST *list;

#ifndef WIN32
// ////////////////////////////////////////////////////////////////////////
void sighandler(int s)
{
	fprintf(stderr, "signal: %d\n", s);
	switch (s)
	{
		case SIGPIPE: 
			fprintf(stderr, "PIPE Quebrado.\n");
			break;

		case SIGSTOP: 
			fprintf(stderr, "STOP\n");
			break;

		default:
			if_thread_list_finalize(list);
			main_alive = TH_ZOMBIE;
			break;
	}
}
// ////////////////////////////////////////////////////////////////////////
#endif


// ////////////////////////////////////////////////////////////////////////
void * th_event(IF_THREAD_LIST *list, IF_THREAD_EVENT *ev)
{
	int n = if_slist_length(list->first);

	switch (ev->event)
	{
		case IF_EVENT_START: fprintf(stdout, "Total:%d  -  EV: start (%d)\n", n, ev->id); break;
		case IF_EVENT_WAIT: fprintf(stdout, "Total:%d  -  EV: wait (%d)\n", n, ev->id); break;
		case IF_EVENT_FREE: fprintf(stdout, "Total:%d  -  EV: FREE (%d)\n", n, ev->id); break;
#ifdef WIN32
		case IF_EVENT_IDLE: Sleep(1); break;
		case IF_EVENT_END: fprintf(stdout, "Total:%d  -  EV: end\n", n); Sleep(1000); break;
#else
		case IF_EVENT_IDLE: usleep(1000); break;
		case IF_EVENT_END: fprintf(stdout, "Total:%d  -  EV: end\n", n); sleep(1); break;
#endif
		default: break;
	}

	return(NULL);
}
// ////////////////////////////////////////////////////////////////////////


// ////////////////////////////////////////////////////////////////////////
void * th_func(THREAD_STATE *state, intptr_t id, void *data)
{
	long int s;

	for (s = random() / (RAND_MAX / 100000) ; s > 30000 ; s = random() / (RAND_MAX / 100000))
		;

	//if_thread_mutex_lock(list);
	if_sleep(s);
	fflush(stdout);
	//if_thread_mutex_unlock(list);

	return(NULL);
}
// ////////////////////////////////////////////////////////////////////////


// ///////////////////////////////////////////////////////////////////// //
int main(int argc, char *argv[])
{
	int i;

#ifndef WIN32 
	signal(SIGINT, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGPIPE, sighandler);
	signal(SIGSTOP, sighandler);
#endif

	main_alive = TH_ALIVE;

	fprintf(stdout, "Inicia (%d).\n", getpid());
	list = if_thread_list_new(th_event);

	for (i = 1 ; i < 300 ; i++)
	{
		if_thread_add(list, th_func, i, NULL);
	}
	
	for (i = 0 ; i < 45 ; i++)
	{
		fprintf(stdout, "Espera (%d)\n", i);
		if_sleep(1000);
	}

	fprintf(stdout, "Finaliza lista.\n");
	if_thread_list_finalize(list);
	fprintf(stdout, "Espera lista.\n");
	i = if_thread_list_wait(list);

	fprintf(stdout, "Encerra (%d). Wait (%d)\n", getpid(), i);
	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
#endif


