#include <ifractal.h>
#include <ifdevice.h>
#include <idx.h>

#include <cidbiolib.h>

#define VENDOR                          "idbio"

// Implementacao //////////////////////////////////////////////////////// //
// ///////////////////////////////////////////////////////////////////// //
int32_t idbio_init()
{
	int32_t ret;
	char *msg = NULL;

	ret = CIDBIO_Init();
	if (ret < CIDBIO_SUCCESS)
	{
		CIDBIO_GetErrorMessage(ret, &msg);
		verbose(stderr, "CIDBIO_Init: %s\n", msg);
		CIDBIO_FreeString(msg);
	}
	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * getInfo(IFDEVICE4J *dev4j, intptr_t sock)
{
	JSON_VALUE *jinfo = NULL;
	char *version, *serial, *model, *val;
	int width = 260, height = 300;

	struct idbio_info_params
	{
		char *name;
		int code;
	}
	param[] = {
		{"variancia_minima", CIDBIO_PARAM_MIN_VAR},
		{"threshold", CIDBIO_PARAM_SIMILIARITY_THRESHOLD},
		{"buzzer", CIDBIO_PARAM_BUZZER_ON},
		{"format", CIDBIO_PARAM_TEMPLATE_FORMAT}
	};
	int32_t i, ret;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(NULL);

	jinfo = json_object_new(1);
	ret = CIDBIO_GetDeviceInfo(&version, &serial, &model);
	if(ret < CIDBIO_SUCCESS)
		goto getInfo_end;

	json_object_add(jinfo, "version", json_string_new(version));
	json_object_add(jinfo, "serial", json_string_new(serial));
	json_object_add(jinfo, "model", json_string_new(model));
	json_object_add(jinfo, "width", json_integer_new(width));
	json_object_add(jinfo, "height", json_integer_new(height));

	CIDBIO_FreeString(version);
	CIDBIO_FreeString(serial);
	CIDBIO_FreeString(model);

	for (i = 0 ; i < sizeof(param) / sizeof(struct idbio_info_params) ; i++)
	{
		CIDBIO_GetParameter(param[i].code, &val);
		json_object_add(jinfo, param[i].name, json_string_new(val));
		CIDBIO_FreeString(val);
	}
getInfo_end:

	CIDBIO_Terminate();

	return(jinfo);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * getUsers(IFDEVICE4J *dev4j, intptr_t ptr)
{
	JSON_VALUE *jusers_ids = json_array_new(1);
	int64_t *ids;
	uint32_t len = 0, ret;
	char msg[PATH_LEN]; 
	
	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(NULL);

	ret = CIDBIO_GetTemplateIDs(&ids, &len);
	if(ret < CIDBIO_SUCCESS)
		goto getUsers_end;

	for(int i = 0; i < len; i++)
	{
		snprintf(msg, PATH_LEN, "%lld", ids[i]);
		JSON_VALUE *juser = json_object_new(1);
		json_object_add(juser, "id", json_string_new(msg));
		json_array_add(jusers_ids, juser);
	}
	
	CIDBIO_FreeIDArray(ids);
getUsers_end:

	CIDBIO_Terminate();

	return(jusers_ids);
}
// ///////////////////////////////////////////////////////////////////// //
int sendBio_iter(JSON_VALUE *juser, void *jres)
{
	char *id = ifdevice_getID(juser);
	int32_t i, ret;
	JSON_VALUE *jtemplates = json_object_find(juser, "templates");
	
	if (id[0] == 0)
	{
		ifdevice_addResult(juser, jres, "ID (id|pis|cracha|matricula|codigo) nao localizado.", 1);
		return(0);
	}

	if (jtemplates == NULL)
                return(0);

	for(i = 0; i != json_array_length(jtemplates); i++)
	{
		char *temp = json_object_get_string((json_array_index(jtemplates, i)), "template");
		char *vendor = json_object_get_string((json_array_index(jtemplates, i)), "vendor");

		if (strcmp(vendor, VENDOR) != 0)
			return(0);

		int64_t pis = strtoll(id, NULL, 0);
		ret = CIDBIO_SaveTemplate(pis, temp);
		if (ret >= CIDBIO_SUCCESS)
			ifdevice_addResult(juser, jres, "Biometria gravada", 0);
		else
			ifdevice_addResult(juser, jres, "Erro ao tentar gravar biometria", 0);
	}

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * sendBio(IFDEVICE4J *dev4j, JSON_VALUE *jusers)
{
	int32_t ret;
	JSON_VALUE *jres = json_array_new(1);

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(jres);

	json_array_iter(jusers, sendBio_iter, jres);

	scanner_file_sendBio(dev4j, jusers, VENDOR);

	CIDBIO_Terminate();

	return(jres);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT deleteBio(SCANNER *scan, char *id, int *qty)
{
	int32_t ret;
	SCANNER_RESULT sr = SCANNER_RESULT_OK;
	char *temp;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);

	int64_t pis = strtoll(id, NULL, 0);
	ret = CIDBIO_GetTemplate(pis, &temp);
	if(ret == CIDBIO_ERROR_NO_TEMPLATE_WITH_ID)
	{
		sr = SCANNER_RESULT_BIO_NOT_ENROLLED;
		goto deleteBio_end;
	}
	CIDBIO_FreeString(temp);

	ret = CIDBIO_DeleteTemplate(pis);
	if(ret < CIDBIO_SUCCESS)
		sr = SCANNER_RESULT_DELETE_ERROR;


deleteBio_end:

	*qty = scanner_file_deleteBio(scan, id);

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
int getBio_iter(JSON_VALUE *juser, void *user_data)
{
	void **ctx = (void **) user_data;
	char *id = ifdevice_getID(juser);
	int32_t ret;
	JSON_VALUE *jtemplates = json_object_find(juser, "templates");
	JSON_VALUE *jres = (JSON_VALUE *) ctx[0];
	int *qtd = (int *) ctx[1];
	char *temp, *msg;
	int finger = 0;

	if (id[0] == 0)
	{
		ifdevice_addResult(juser, jres, "ID (id|pis|cracha|matricula|codigo) nao localizado.", 1);
		return(0);
	}

	if (jtemplates == NULL)
	{
		jtemplates = json_array_new(1);
		json_object_add(juser, "templates", jtemplates);
	}

	int64_t pis = strtoll(id, NULL, 0);
	ret = CIDBIO_GetTemplate(pis, &temp);
	if (ret < CIDBIO_SUCCESS)
	{
		CIDBIO_GetErrorMessage(ret, &msg);
		verbose(stderr, "CIDBIO_GetTemplate: %s\n", msg);
		CIDBIO_FreeString(msg);
		return(0);
	}

	JSON_VALUE *jtemp = json_object_new(1);
	json_array_add(jtemplates, jtemp);

	json_object_add(jtemp, "template", json_string_new(temp));
	json_object_add(jtemp, "vendor", json_string_new(VENDOR));
	json_object_add(jtemp, "finger", json_integer_new(finger));
	json_object_add(jtemp, "encode", json_string_new(TXT_ENCODE_B64));

	CIDBIO_FreeString(temp);

	qtd += 1;

	return(0);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * getBio(IFDEVICE4J *dev4j, JSON_VALUE *users)
{
	int32_t ret;
	JSON_VALUE *jres = NULL;
	int qtd = 0;
	void *ctx[] = {jres, &qtd};

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(NULL);

	json_array_iter(users, getBio_iter, ctx);

	CIDBIO_Terminate();

	return(users);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT captureImage(SCANNER *scan, int qty, void *user_data)
{
	uint32_t width, height, ret;
	uint8_t *imgBuf = NULL;
	SCANNER_RESULT sr = SCANNER_RESULT_OK;
	SCANNER_MESSAGE_CALLBACK msg_cb = scan->message_callback;
	SCANNER_IMAGE_CALLBACK img_cb = scan->image_callback;
	int i;
	char *msg;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);

	for (i = 0; i < qty; i++)
	{	
		msg_cb(scan, VENDOR, "Posicione o dedo no sensor...", user_data);

		ret = CIDBIO_CaptureImage(&imgBuf, &width, &height);
		
		if (ret >= CIDBIO_SUCCESS)
		{
			img_cb(scan, imgBuf, width, height, user_data);
			CIDBIO_FreeByteArray(imgBuf);
		}
	}

	if (ret < CIDBIO_SUCCESS)
	{
		CIDBIO_GetErrorMessage(ret, &msg);
		verbose(stderr, "CIDBIO_CaptureImage: %s\n", msg);
		CIDBIO_FreeString(msg);
		sr = SCANNER_RESULT_CAPTURE_ERROR;
	}

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT enroll(SCANNER *scan, int qty_frames, char *id, int8_t *quality, void *user_data)
{
	char *temp;
	char *msg;
	uint8_t *imgBuf = NULL;
	uint32_t width, height, ret;
        int32_t q;
	SCANNER_RESULT sr = SCANNER_RESULT_OK;
	SCANNER_MESSAGE_CALLBACK msg_cb = scan->message_callback;
	SCANNER_IMAGE_CALLBACK img_cb = scan->image_callback;

	if (id == NULL)
		return(SCANNER_RESULT_INVALID_PARAMETERS);

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);
	
	msg_cb(scan, VENDOR, "Posicione o dedo no sensor...", user_data);

	ret = CIDBIO_CaptureImageAndTemplate((char **const)&temp, &imgBuf, &width, &height, &q);
	if (ret < CIDBIO_SUCCESS)
	{
		CIDBIO_GetErrorMessage(ret, &msg);
		verbose(stderr, "CIDBIO_CaptureImageAndTemplate: %s\n", msg);
		CIDBIO_FreeString(msg);
		sr = SCANNER_RESULT_CAPTURE_ERROR;
		goto enroll_end;
	}

	img_cb(scan, imgBuf, width, height, user_data);
	CIDBIO_FreeByteArray(imgBuf);

	*quality = (int8_t) q/10;
	scan->template_callback(scan, VENDOR, (uint8_t *)temp, strlen(temp), *quality, user_data);

	int64_t pis = strtoll(id, NULL, 0);
	ret = CIDBIO_SaveTemplate(pis, temp);
	if (ret < CIDBIO_SUCCESS)
		sr = SCANNER_RESULT_ENROLL_DB_ERROR; 

	CIDBIO_FreeString(temp);

enroll_end: 

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT identify(SCANNER *scan, char *id, size_t id_len, int *finger, int32_t *score, void *user_data)
{
	int32_t ret; 
	SCANNER_RESULT sr = SCANNER_RESULT_BIO_NOT_FOUND;
	SCANNER_MESSAGE_CALLBACK msg_cb = scan->message_callback;
	int32_t quality = 0;
	int64_t pis, *ids;
	uint32_t qty_ids;

	*finger = 0;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);

	msg_cb(scan, VENDOR, "Posicione o dedo no sensor...", user_data);

	ret = CIDBIO_CaptureAndIdentify(&pis, score, &quality);
	if(ret < CIDBIO_SUCCESS)
	{
		sr = SCANNER_RESULT_IDENTIFY_ERROR;
		goto identify_end;
	}

	*score /= 20;
	if(*score < 800)
		goto identify_end;
//	Corrige falha na API do CIDBIO
	ret = CIDBIO_GetTemplateIDs(&ids, &qty_ids);
	if(ret < CIDBIO_SUCCESS)
		goto identify_end;

	for(int i = 0; i < qty_ids; i++)
	{
		if((ids[i] & 0xFFFFFFFF) == pis)
		{
			*finger = 1;
			sr = SCANNER_RESULT_OK;
			snprintf(id, id_len, "%lld", ids[i]);
			break;
		}
	}

	CIDBIO_FreeIDArray(ids);

identify_end:

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT verify(SCANNER *scan, char *id, int *result_finger, void *user_data)
{
	int32_t ret; 
	SCANNER_RESULT sr = SCANNER_RESULT_OK;
	int32_t score = 0, quality = 0;
	SCANNER_MESSAGE_CALLBACK msg_cb = scan->message_callback;
	char *temp1, *temp2;
	uint32_t width, height;
	uint8_t *imgBuf;
	SCANNER_IMAGE_CALLBACK img_cb = scan->image_callback;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);

	msg_cb(scan, VENDOR, "Posicione o dedo no sensor...", user_data);

	ret = CIDBIO_CaptureImageAndTemplate(&temp1, &imgBuf, &width, &height, &quality);
	if(ret < CIDBIO_SUCCESS)
	{
		sr = SCANNER_RESULT_CAPTURE_ERROR;
		goto verify_end;
	}

	img_cb(scan, imgBuf, width, height, user_data);

	int64_t pis = strtoll(id, NULL, 0);
	ret = CIDBIO_GetTemplate(pis, &temp2);
	if(ret == CIDBIO_ERROR_NO_TEMPLATE_WITH_ID)
	{
		sr = SCANNER_RESULT_BIO_NOT_ENROLLED;
		goto verify_end;
	}
	else if(ret < CIDBIO_SUCCESS)
	{
		sr = SCANNER_RESULT_IDENTIFY_DB_ERROR;
		goto verify_end;
	}

	ret = CIDBIO_MatchTemplates(temp1, temp2, &score);
	if (ret < CIDBIO_SUCCESS)
		sr = SCANNER_RESULT_VERIFY_ERROR;

	score = score / 20;

	if(score < 800)
		sr = SCANNER_RESULT_BIO_NOT_FOUND;
verify_end:

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT delete_all()
{
	int32_t ret; 
	SCANNER_RESULT sr = SCANNER_RESULT_OK;

	ret = idbio_init();
	if (ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_MODULE_ERROR);

	ret = CIDBIO_DeleteAllTemplates();
	if(ret < CIDBIO_SUCCESS)
		return(SCANNER_RESULT_DELETE_ERROR);

	CIDBIO_Terminate();

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT update_firmware(char *filePath)
{
	int32_t ret; 
	char *msg = NULL;
	SCANNER_RESULT sr = SCANNER_RESULT_OK;

	ret = CIDBIO_UpdateFirmware(filePath);
	if(ret < CIDBIO_SUCCESS)
	{
		sr = SCANNER_RESULT_UP_FIRMWARE_ERROR;
		CIDBIO_GetErrorMessage(ret, &msg);
		printf("CIDBIO_UpdateFirmware: %s\n", msg);
		CIDBIO_FreeString(msg);
	}

	return(sr);
}
// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT finalize(SCANNER *scan)
{
	return(0);
}

// ///////////////////////////////////////////////////////////////////// //
SCANNER_RESULT ifdevice_init(SCANNER *scan)
{
	SCANNER_RESULT ret = 0;

	scan->dev.getInfo = getInfo;
	scan->dev.sendBio = sendBio;
	scan->dev.getBio = getBio;
	scan->dev.getUsers = getUsers;
	scan->deleteBio = deleteBio;
	scan->captureImage = captureImage;
	scan->enroll = enroll;
	scan->identify = identify;
	scan->verify = verify;
	scan->delete_all = delete_all;
	scan->update_firmware = update_firmware;
	scan->finalize = finalize;

	return(ret);
}

// ///////////////////////////////////////////////////////////////////// //
