#include <ifractal.h>
#include <boleto_barras.h>


TAG_BOLETO tags_bb_semreg[] = {
	{"banco", "001", 3},
	{"agencia", NULL, 4},
	{"conta", NULL, 5},
	{"nosso_nro", NULL, 11},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_bb[] = {
	{"banco", "001", 3},
	{"convenio", NULL, 6},
	{"nosso_nro", NULL, 17},
	{"carteira", NULL, 2},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_bradesco[] = {
	{"banco", "237", 3},
	{"agencia", NULL, 4},
	{"conta", NULL, 7},
	{"nosso_nro", NULL, 11},
	{"carteira", NULL, 2},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_santander[] = {
	{"banco", "033", 3},
	{"cod_cedente", NULL, 7},
	{"nosso_nro", NULL, 12},
	{"carteira", NULL, 3},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_itau[] = {
	{"banco", "341", 3},
	{"agencia", NULL, 4},
	{"cod_cedente", NULL, 6},
	{"nosso_nro", NULL, 8},
	{"carteira", NULL, 3},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_itau198[] = {
	{"banco", "341", 3},
	{"carteira", NULL, 3},
	{"nosso_nro", NULL, 8},
	{"nro_documento", NULL, 7},
	{"cod_cedente", NULL, 5},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_panamericano[] = {
	{"banco", "623", 3},
	{"agencia", NULL, 4},
	{"carteira", NULL, 3},
	{"cod_cedente", NULL, 7},
	{"nosso_nro", NULL, 10},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_citi[] = {
	{"banco", "745", 3},
	{"carteira", NULL, 3},
	{"cod_cedente", NULL, 9},
	{"nosso_nro", NULL, 11},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_caixa[] = {
	{"banco", "104", 3},
	{"agencia", NULL, 4},
	{"cod_cedente", NULL, 11},
	{"nosso_nro", NULL, 10},
	{"vencimento", NULL},
	{"valor", NULL},
	{NULL, NULL}
};

TAG_BOLETO tags_cef_sigcb[] = {
	{"banco", "104", 3},
	{"cod_cedente", NULL, 6},
	{"nosso_nro", NULL, 15},
	{"carteira", NULL, 1},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_bgn[] = {
	{"banco", "739", 3},
	{"carteira", "30", 2},
	{"cod_cedente", NULL, 3},
	{"nosso_nro", NULL, 13},
	{"parcela", "000", 3},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_real[] = {
	{"banco", "356", 3},
	{"agencia", NULL, 4},
	{"cod_cedente", NULL, 7},
	{"nosso_nro", NULL, 13},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};

TAG_BOLETO tags_hsbc[] = {
	{"banco", "399", 3},
	{"cod_cedente", NULL, 7},
	{"nosso_nro", NULL, 13},
	//{"data_juliano", "0000", 4},
	//{"cnr", "2", 1},
	{"vencimento", NULL, 0},
	{"valor", NULL, 0},
	{NULL, NULL}
};


// ///////////////////////////////////////////////////////////////////// //
char * barras_get_value(TAG_BOLETO tags[], char *name)
{
	char *value = NULL;
	int i;

	for (i = 0 ; tags[i].tag != NULL ; i++)
	{
		if (strcmp(tags[i].tag, name) == 0)
		{
			value = tags[i].value;
			break;
		}
	}

	return(value);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void barras_set_value(TAG_BOLETO tags[], char *name, char *value)
{
	int i;

	for (i = 0 ; tags[i].tag != NULL ; i++)
	{
		if (strcmp(tags[i].tag, name) == 0)
		{
			tags[i].value = value;
			return;
		}
	}
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int fill_tags(
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *params,
	_IN TAG_BOLETO tags[])
{
	JSON_VALUE *value, *globals;
	char *tag;
	int i;

	globals = json_object_find(result, "globals");
	for (i = 0 ; tags[i].tag != NULL ; i++)
	{
		tag = param_get_value(params, tags[i].tag);
		value = json_object_find(record, tag);
		if (value == NULL)
			value = json_object_find(globals, tag);

		if (value == NULL)
		{
			//addException(result, tag, data_source, 0);
			//addException(result, "Erro ao tentar gerar barras.", data_source, 0);
			continue;
		}

		tags[i].value = json_get_string(value);
	}

	return(i);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void parser_barras(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params,
	_IN TAG_BOLETO tags[],
	_IN void (*calc_barras)(char *))
{
	char *tag_ld, ld[BARRAS_LEN];

	fill_tags(result, record, data_source, params, tags);

	calc_barras(data);
	calc_linha_digitavel(data, ld);

	tag_ld = param_get_value(params, "linha");
	json_object_add(record, tag_ld, json_string_new(ld));
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * parser_santander(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_santander, calc_barras_santander);
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * parser_bb(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_bb, calc_barras_bb);
	return(NULL);
}
// ///////////////////////////////////////////////////////////////////// //
JSON_VALUE * parser_bradesco(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_bradesco, calc_barras_bradesco);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_caixa(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_caixa, calc_barras_caixa);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_itau(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_itau, calc_barras_itau);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_itau198(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_itau198, calc_barras_itau198);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_panamericano(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_panamericano, calc_barras_panamericano);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_citi(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_citi, calc_barras_citi);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_cef_sigcb(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_cef_sigcb, calc_barras_cef_sigcb);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_bgn(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_bgn, calc_barras_bgn);
	return(NULL);
}
// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_real(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	parser_barras(data, result, record, templ, data_source, token, params, tags_real, calc_barras_real);
	return(NULL);
}
// ////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////
JSON_VALUE * parser_linha2barras(
	_INOUT char *data,
	_IN JSON_VALUE *result, 
	_IN JSON_VALUE *record, 
	_IN JSON_VALUE *templ, 
	_IN JSON_VALUE *data_source, 
	_IN JSON_VALUE *token,
	_IN JSON_VALUE *params)
{
	char linha[BARRAS_LEN], *barras, *tag_barras;

	strncpy(linha, data, BARRAS_LEN);

	tag_barras = param_get_value(params, "codigo_barras");
	barras = linha2barras(linha);
	json_object_add(record, tag_barras, json_string_new(barras));
	if_free(barras);

	return(NULL);
}
// ////////////////////////////////////////////////////////////



// ///////////////////////////////////////////////////////////////////// //
void calc_fator(_INOUT char *barras, _IN char *data)
{
	struct tm ref;
	struct tm venc;
	time_t unix_ref, unix_venc;
	time_t fator;

	if ((data == NULL) || (strlen(data) < 10))
	{
		fprintf(stderr, "Erro ao tentar calcular fator...\n");
		sprintf(barras, "0000");
		return;
	}

	// 07 de Outubro de 1997
	memset(&ref, 0, sizeof(struct tm));
	ref.tm_mday = 7;
	ref.tm_mon = 10 - 1;
	ref.tm_year = 97;

	memset(&venc, 0, sizeof(struct tm));
	venc.tm_mday = 10 * (data[0] - '0') + (data[1] - '0');
	venc.tm_mon = (10 * (data[3] - '0') + (data[4] - '0')) - 1;
	venc.tm_year = 100 + 10 * (data[8] - '0') + (data[9] - '0');

	unix_ref = mktime(&ref);
	unix_venc = mktime(&venc);

	fator = (unix_venc - unix_ref) / 86400;
	if (fator > 9999)
		fator -= 9000;

	sprintf(barras, "%04ld\n", fator);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
int calc_valor(_IN char *v)
{
	int ret;
	int i, decimais, d;

	if (v == NULL)
		return(0);

	for (ret = 0, d = 0, i = 0, decimais = 2 ; v[i] != 0 ; i++)
	{
		if (v[i] == ',')
		{
			d = 1;
			continue;
		}

		if ((v[i] > '9') || (v[i] < '0'))
			continue;

		if (d)
			decimais--;

		ret = 10 * ret + (v[i] - '0');
	}

	for (i = 0 ; i < decimais ; i++)
		ret *= 10;

	return(ret);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
// Modulo 11
int calc_DAC11_in(char *br, int i, int f)
{
	int soma, k, r, dig, resto;

	if (br == NULL)
		return(' ');

	k = strlen(br);
	if ((k < f) || (i >= f))
		return(' ');

	for (r = 2, k = f - 1, soma = 0 ; k >= i ; k--, r++)
	{
		if (r > 9)
			r = 2;

		dig = br[k] - '0';
		soma += r * dig;
	}

	resto = soma % 11;

	return(resto);
}
// ///////////////////////////////////////////////////////////////////// //
char calc_DAC11_citi(char *br, int i, int f)
{
	char ret;
	int resto = calc_DAC11_in(br, i, f);

	if ((resto == 0) || (resto == 1))
		ret = 0;
	else
		ret = (char) (11 - resto);
	
	return(ret + '0');
}
// ///////////////////////////////////////////////////////////////////// //
char calc_DAC11_santander(char *br, int i, int f)
{
	return(calc_DAC11_citi(br, i, f));
}
// ///////////////////////////////////////////////////////////////////// //
char calc_DAC11_bradesco(char *br, int i, int f)
{
	int soma, k, r, dig, resto;
	char dac;

	if (br == NULL)
		return(' ');

	k = strlen(br);
	if ((k < f) || (i >= f))
		return(' ');

	for (r = 2, k = i, soma = 0 ; k < f ; k++, r--)
	{
		if (r < 2)
			r = 7;

		dig = br[k] - '0';
		soma += r * dig;
	}

	resto = soma % 11;

	switch (resto)
	{
		case 0: dac = '0'; break;
		case 1: dac = 'P'; break;
		default: dac = '0' + (11 - resto);
	}

	return(dac);
}
// ///////////////////////////////////////////////////////////////////// //
char calc_DAC11(char *br, int i, int f)
{
	char ret;
	int resto = calc_DAC11_in(br, i, f);

	if ((resto == 0) || (resto == 10) || (resto == 1))
		ret = 1;
	else
		ret = (char) (11 - resto);
	
	return(ret + '0');
}
// ///////////////////////////////////////////////////////////////////// //
	
// ///////////////////////////////////////////////////////////////////// //
// Modulo 10
char calc_DAC10(char *ln, int i, int f)
{
	int k, r, dig;
	int soma;
	char ret;

	for (r = 0, k = f - 1, soma = 0 ; k >= i ; k--, r++)
	{
		dig = ln[k] - '0';
		if ((r % 2) == 0)
		{
			dig *= 2;
			if (dig > 9)
				dig = (dig - 10) + 1;
		}

		soma += dig;
	}

	ret = (char) (10 - (soma % 10));
	if (ret > 9)
		ret = 0;

	return(ret + '0');
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
char calc_DAC11_cef(char *br, int i, int f)
{
	int soma, k, r, dig, resto, sub;
	char ret;

	if (br == NULL)
		return(' ');

	k = strlen(br);
	if ((k < f) || (i >= f))
		return(' ');

	for (r = 2, k = f - 1, soma = 0 ; k >= i ; k--, r++)
	{
		if (r > 9)
			r = 2;

		dig = br[k] - '0';
		soma += r * dig;
	}

	resto = soma % 11;
	sub = 11 - resto;
	if (sub > 9)
		ret = 0;
	else
		ret = sub;
	
	return(ret + '0');
}
// ///////////////////////////////////////////////////////////////////// //
	

// ///////////////////////////////////////////////////////////////////// //
_PRIVATE int barras_vencimento_valor(_INOUT char *barras, _IN TAG_BOLETO tags[])
{
	char *banco = barras_get_value(tags, "banco");
	char *vencimento = barras_get_value(tags, "vencimento");
	char *str_valor = barras_get_value(tags, "valor");
	int i, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	i = snprintf(barras, BARRAS_LEN, "%s9", banco);

	memset(barras + i, '0', 14);

	if ((vencimento != NULL) && (strlen(vencimento) == 10))
		calc_fator(barras + i, vencimento);

	i += 4;
	if (str_valor != NULL)
	{
		valor = calc_valor(str_valor);
		sprintf(barras + i, "%010d", valor);
	}

	return(i + 10);
}
// ///////////////////////////////////////////////////////////////////// //



// ///////////////////////////////////////////////////////////////////// //
void calc_linha_digitavel(_IN char *barras, _INOUT char *ld)
{
	char dac, bloco[BARRAS_LEN];

	// Monta linha digitavel
	// bloco 1
	memset(ld, 0, BARRAS_LEN);
	memcpy(ld, barras, 4);
	ld[4] = barras[19];
	ld[5] = '.';
	memcpy(ld + 6, barras + 20, 4);

	memcpy(bloco, ld, 5);
	memcpy(bloco + 5, ld + 6, 4);
	dac = calc_DAC10(bloco, 0, 9);
	ld[10] = dac;

	// bloco 2
	ld[11] = ' ';
	memcpy(ld + 12, barras + 24, 5);
	ld[17] = '.';
	memcpy(ld + 18, barras + 29, 5);

	memcpy(bloco, ld + 12, 5);
	memcpy(bloco + 5, ld + 18, 6);
	dac = calc_DAC10(bloco, 0, 10);
	ld[23] = dac;

	// bloco 3
	ld[24] = ' ';
	memcpy(ld + 25, barras + 34, 5);
	ld[30] = '.';
	memcpy(ld + 31, barras + 39, 5);

	memcpy(bloco, ld + 25, 5);
	memcpy(bloco + 5, ld + 31, 5);
	dac = calc_DAC10(bloco, 0, 10);
	SNPRINTF(ld + 36, BARRAS_LEN, "%c %c ", dac, barras[4]);

	memcpy(ld + 40, barras + 5, 14);
	barras[55] = 0;
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void finaliza_barras(_INOUT char *barras)
{
	char *p, *q, dac;

	dac = calc_DAC11(barras, 0, 43);
	for (p = barras + BARRAS_LEN - 1, q = p - 1 ; (p - barras) > 4 ; *p-- = *q--);
	barras[4] = dac;
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void calc_barras_bb_semreg(_INOUT char *barras)
{
	char *agencia = barras_get_value(tags_bb_semreg, "agencia");
	char *conta = barras_get_value(tags_bb_semreg, "conta");
	char *nn = barras_get_value(tags_bb_semreg, "nosso_nro");
	char *vencimento = barras_get_value(tags_bb_semreg, "vencimento");
	char *str_valor = barras_get_value(tags_bb_semreg, "valor");
	int i, len, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "0019");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	//memset(barras + 4, '0', 14);

	len = 11;
	memcpy(barras + i, nn, len);

	i += len;
	len = 4;
	memcpy(barras + i, agencia, len);

	i += len;
	len = 3;
	memset(barras + i, '0', len);

	i += len;
	len = 5;
	memcpy(barras + i, conta, len);

	i += len;
	SNPRINTF(barras + i, BARRAS_LEN, "17");

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //
void calc_barras_bb(_INOUT char *barras)
{
	char *convenio = barras_get_value(tags_bb, "convenio");
	char *nn = barras_get_value(tags_bb, "nosso_nro");
	char *carteira = barras_get_value(tags_bb, "carteira");
	char *vencimento = barras_get_value(tags_bb, "vencimento");
	char *str_valor = barras_get_value(tags_bb, "valor");
	int i, len, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "0019");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	len = 6;
	memcpy(barras + i, convenio, len);
	i += len;
	len = 17;
	memcpy(barras + i, nn, len);
	i += len;
	len = 2;
	memcpy(barras + i, carteira, len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //



// ///////////////////////////////////////////////////////////////////// //
void calc_barras_santander(_INOUT char *barras)
{
	char *cod_cedente = barras_get_value(tags_santander, "cod_cedente");
	char *nn = barras_get_value(tags_santander, "nosso_nro");
	char *carteira = barras_get_value(tags_santander, "carteira");
	char *vencimento = barras_get_value(tags_santander, "vencimento");
	char *str_valor = barras_get_value(tags_santander, "valor");
	int len, i, valor;
	char dac;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "0339");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	len = 1;
	memcpy(barras + i, "9", len);
	i += len;
	len = 7;
	memcpy(barras + i, cod_cedente, len);
	i += len;
	len = 12;
	memcpy(barras + i, nn, len);
	dac = calc_DAC11_santander(barras, i, i + len);
	i += len;
	barras[i++] = dac;
	len = 1;
	memcpy(barras + i, "0", len);
	i += len;
	len = 3;
	memcpy(barras + i, carteira, len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void calc_barras_bradesco(_INOUT char *barras)
{
	char *agencia = barras_get_value(tags_bradesco, "agencia");
	char *conta = barras_get_value(tags_bradesco, "conta");
	char *carteira = barras_get_value(tags_bradesco, "carteira");
	char *nn = barras_get_value(tags_bradesco, "nosso_nro");
	char *vencimento = barras_get_value(tags_bradesco, "vencimento");
	char *str_valor = barras_get_value(tags_bradesco, "valor");
	int len, i, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "2379");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	len = 4;
	memcpy(barras + i, agencia, len);
	i += len;
	len = 2;
	memcpy(barras + i, carteira, len);
	i += len;
	len = 11;
	memcpy(barras + i, nn, len);
	i += len;
	len = 7;
	memcpy(barras + i, conta, len);
	i += len;
	len = 1;
	memcpy(barras + i, "0", len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
void calc_barras_caixa(_INOUT char *barras)
{
	char *agencia = barras_get_value(tags_caixa, "agencia");
	char *cod_cedente = barras_get_value(tags_caixa, "cod_cedente");
	char *nn = barras_get_value(tags_caixa, "nosso_nro");
	char *vencimento = barras_get_value(tags_caixa, "vencimento");
	char *str_valor = barras_get_value(tags_caixa, "valor");
	int len, i, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "1049");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	len = 10;
	memcpy(barras + i, nn, len);
	i += len;
	len = 4;
	memcpy(barras + i, agencia, len);
	i += len;
	len = 11;
	memcpy(barras + i, cod_cedente, len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //

// ///////////////////////////////////////////////////////////////////// //
char calc_itau_dac(char *agencia, char *cod_cedente, char *carteira, char *nn)
{
	char *esc[] = {"112", "126", "131", "145", "150", "168", NULL};
	int daclen = 4 + 5 + 3 + 8 + 1;	// Agencia + cod_cedente + carteira + nosso nro + finalizador 0x0
	char auxdac[daclen];
	int i, l, offset = 0;
	char dac;

	i = 0;
	l = 4;
	memcpy(auxdac + i, agencia, l);
	i += l;
	l = 5;
	memcpy(auxdac + i, cod_cedente, l);
	i += l;
	l = 3;
	memcpy(auxdac + i, carteira, l);
	i += l;
	l = 8;
	memcpy(auxdac + i, nn, l);
	i += l;
	auxdac[i] = 0;

	for (i = 0 ; esc[i] != NULL ; i++)
		if (strncmp(carteira, esc[i], 3) == 0)
			offset = 9;

	dac = calc_DAC10(auxdac + offset, 0, daclen - 1 - offset);

	return(dac);
}
// ///////////////////////////////////////////////////////////////////// //
void calc_barras_itau(_INOUT char *barras)
{
	char *agencia = barras_get_value(tags_itau, "agencia");
	char *cod_cedente = barras_get_value(tags_itau, "cod_cedente");
	char *nn = barras_get_value(tags_itau, "nosso_nro");
	char *carteira = barras_get_value(tags_itau, "carteira");
	char *vencimento = barras_get_value(tags_itau, "vencimento");
	char *str_valor = barras_get_value(tags_itau, "valor");
	int len, i, valor;

	// Monta codigo de barras
	memset(barras, 0, BARRAS_LEN);
	sprintf(barras, "3419");

	i = 4;
	calc_fator(barras + i, vencimento);

	i += 4;
	memset(barras + i, '0', 10);

	valor = calc_valor(str_valor);
	sprintf(barras + i, "%010d", valor);

	i += 10;
	len = 3;
	memcpy(barras + i, carteira, len);
	i += len;
	len = 8;
	memcpy(barras + i, nn, len);
	i += len;
	*(barras + i) = calc_itau_dac(agencia, cod_cedente, carteira, nn);
	i++;
	len = 4;
	memcpy(barras + i, agencia, len);
	i += len;
	len = 6;
	memcpy(barras + i, cod_cedente, len);
	i += len;
	len = 3;
	memcpy(barras + i, "000", len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
void calc_barras_itau198(_INOUT char *barras)
{
	int len, i, k, str_dac_len;

	i = barras_vencimento_valor(barras, tags_itau198);

	for (k = 1, str_dac_len = 0 ; tags_itau198[k].len > 0 ; k++)
	{
		len = tags_itau198[k].len;
		str_dac_len += len; 
		memcpy(barras + i, tags_itau198[k].value, len);
		i += len;
	}

	*(barras + i) = calc_DAC10(barras, 18, 18 + str_dac_len);
	i++;
	len = 1;
	memcpy(barras + i, "0", len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PRIVATE char calc_panamericano_dac()
{
	int dac_len, len, i, k, tags[] = {1,2,4,-1};	// agencia, carteira, nosso_nro
	char buf[PATH_LEN];

	for (i = 0, k = 0, dac_len = 0 ; tags[k] > 0 ; k++)
	{
		len = tags_panamericano[tags[k]].len;
		memcpy(buf + i, tags_panamericano[tags[k]].value, len);
		i += len;
		dac_len += len;
	}

	return(calc_DAC10(buf, 0, dac_len));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC void calc_barras_panamericano(_INOUT char *barras)
{
	int len, i, k;

	i = barras_vencimento_valor(barras, tags_panamericano);

	for (k = 1 ; tags_panamericano[k].len > 0 ; k++)
	{
		len = tags_panamericano[k].len;
		memcpy(barras + i, tags_panamericano[k].value, len);
		i += len;
	}

	*(barras + i) = calc_panamericano_dac();

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC void calc_barras_citi(_INOUT char *barras)
{
	int len, i, k;

	i = barras_vencimento_valor(barras, tags_citi);

	memcpy(barras + i, "3", 1);
	for (k = 1, i++ ; tags_citi[k].len > 0 ; k++)
	{
		len = tags_citi[k].len;
		memcpy(barras + i, tags_citi[k].value, len);
		i += len;
	}

	*(barras + i) = calc_DAC11_citi(tags_citi[--k].value, 0, 11);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC void calc_barras_cef_sigcb(_INOUT char *barras)
{
	int len, i, inicio_campolivre;

	i = barras_vencimento_valor(barras, tags_cef_sigcb);
	inicio_campolivre = i;

	// cedente
	len = tags_cef_sigcb[1].len;
	memcpy(barras + i, tags_cef_sigcb[1].value, len);
	i += len;

	*(barras + i) = calc_DAC11_cef(tags_cef_sigcb[1].value, 0, tags_cef_sigcb[1].len);
	i++;
	
	// nosso_nro - primeira parte
	len = 3;
	memcpy(barras + i, tags_cef_sigcb[2].value, len);
	i += len;

	// carteira (1 registrada - 2 sem registro)
	*(barras + i) = tags_cef_sigcb[3].value[0];
	i++;
	
	// nosso_nro - segunda parte
	len = 3;
	memcpy(barras + i, tags_cef_sigcb[2].value + 3, len);
	i += len;

	// emitido pelo cliente
	*(barras + i) = '4';
	i++;
	
	// nosso_nro - terceira parte
	len = 9;
	memcpy(barras + i, tags_cef_sigcb[2].value + 6, len);
	i += len;

	*(barras + i) = calc_DAC11_cef(barras, inicio_campolivre, inicio_campolivre + 24);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC void calc_barras_bgn(_INOUT char *barras)
{
	int len, i, k;

	i = barras_vencimento_valor(barras, tags_bgn);

	for (k = 1 ; tags_bgn[k].len > 0 ; k++)
	{
		len = tags_bgn[k].len;
		memcpy(barras + i, tags_bgn[k].value, len);
		i += len;
	}

	memcpy(barras + i, "0000", 4);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PRIVATE char calc_DAC_real()
{
	int seq[] = {3,1,2,0};
	int size, i, p;

	for (i = 0, size = 1 ; seq[i] != 0 ; i++)
		size += tags_real[seq[i]].len;

	char aux[size];

	for (i = 0, p = 0 ; seq[i] != 0 ; i++)
	{
		memcpy(aux + p, tags_real[seq[i]].value, tags_real[seq[i]].len);
		p += tags_real[seq[i]].len;
	}

	aux[p] = 0;

	return(calc_DAC10(aux, 0, p));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC void calc_barras_real(_INOUT char *barras)
{
	int len, i, k = 1;

	i = barras_vencimento_valor(barras, tags_real);

	len = tags_real[k].len;
	memcpy(barras + i, tags_real[k++].value, len);
	i += len;

	len = tags_real[k].len;
	memcpy(barras + i, tags_real[k++].value, len);
	i += len;

	*(barras + i) = calc_DAC_real();
	i++;

	len = tags_real[k].len;
	memcpy(barras + i, tags_real[k++].value, len);

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
int calc_juliano_hsbc(char *venc)
{
	int mdays[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	int day = atoi(venc);
	int mon = atoi(venc + 3);
	int yr = atoi(venc + 6);
	int i, jul = 0;

	if (((yr % 4) == 0) && (mon > 2))
		jul = 1;

	for (i = 0 ; i < (mon - 1) ; i++)
		jul += mdays[i];

	return(jul + day);
}
// ///////////////////////////////////////////////////////////////////// //
void calc_barras_hsbc(_INOUT char *barras)
{
	char *vencimento = barras_get_value(tags_hsbc, "vencimento");
	int jul, len, i, k;

	i = barras_vencimento_valor(barras, tags_hsbc);

	for (k = 1 ; tags_hsbc[k].len > 0 ; k++)
	{
		len = tags_hsbc[k].len;
		memcpy(barras + i, tags_hsbc[k].value, len);
		i += len;
	}

	jul = calc_juliano_hsbc(vencimento);
	i += snprintf(barras + i, 4, "%03d", jul);
	barras[i++] = vencimento[9];
	barras[i] = '2';

	finaliza_barras(barras);
}
// ///////////////////////////////////////////////////////////////////// //



typedef struct
{
	char *name;
	TAG_BOLETO *boleto;
	void (*calc)(char *);
} BOLETO_BANCO;


BOLETO_BANCO bancos[] = {
	{"BB sem registro", tags_bb_semreg, calc_barras_bb_semreg},
	{"Banco do Brasil", tags_bb, calc_barras_bb},
	{"Bradesco", tags_bradesco, calc_barras_bradesco},
	{"Santander", tags_santander, calc_barras_santander},
	{"Caixa Economica Federal", tags_caixa, calc_barras_caixa},
	{"Itau", tags_itau, calc_barras_itau},
	{"Itau carteira 198", tags_itau198, calc_barras_itau198},
	{"Panamericano", tags_panamericano, calc_barras_panamericano},
	{"Citibank", tags_citi, calc_barras_citi},
	{"CEF sigcb", tags_cef_sigcb, calc_barras_cef_sigcb},
	{"BGN", tags_bgn, calc_barras_bgn},
	{"Real", tags_real, calc_barras_real},
	{"HSBC", tags_hsbc, calc_barras_hsbc},
	{NULL, NULL}
};

// ////////////////////////////////////////////////////////////////////////
_PRIVATE int barras_make(char *barras, int argc, char **argv)
{
	int i, banco_i;

	banco_i = atoi(argv[1]) - 1;
	if (banco_i < 0)
	{
		fprintf(stderr, "(1) Indice banco invalido: %s\n", argv[1]);
		return(-1);
	}

	for (i = 0 ; (bancos[i].name != NULL) && (i != banco_i) ; i++)
		;
	
	if (banco_i != i)
	{
		fprintf(stderr, "(2) Indice banco invalido: %d\n", banco_i);
		return(-2);
	}

	for (i = 1 ; bancos[banco_i].boleto[i].tag != NULL ; i++)
	{
		if (argc <= (i + 1))
		{
			fprintf(stderr, "Faltam campos.\n"); 
			return(-3);
		}

		if ((bancos[banco_i].boleto[i].len > 0) && (strlen(argv[1 + i]) != bancos[banco_i].boleto[i].len))
		{
			fprintf(stderr, "Campo '%s': tamanho invalido, deveria ter %ld.\n", 
				bancos[banco_i].boleto[i].tag,
				bancos[banco_i].boleto[i].len);
			return(-4);
		}

		barras_set_value(
			bancos[banco_i].boleto, 
			bancos[banco_i].boleto[i].tag, 
			argv[1 + i]);
	}

	bancos[banco_i].calc(barras);

	return(0);
}
// ////////////////////////////////////////////////////////////////////////



// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_bb_sem_registro(char *agencia, char *conta, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "1", agencia, conta, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_bb2(char *convenio, char *nosso_nro, char *carteira, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "2", convenio, nosso_nro, carteira, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_bb(char *nosso_nro, char *carteira, char *vencimento, char *valor)
{
	return(barras_bb2("000000", nosso_nro, carteira, vencimento, valor));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_bradesco(char *agencia, char *conta, char *nosso_nro, char *carteira, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "3", agencia, conta, nosso_nro, carteira, vencimento, valor, NULL};

	barras_make(barras, 8, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_santander(char *cod_cedente, char *nosso_nro, char *carteira, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "4", cod_cedente, nosso_nro, carteira, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_cef(char *agencia, char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "5", agencia, cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_itau(char *agencia, char *cod_cedente, char *nosso_nro, char *carteira, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "6", agencia, cod_cedente, nosso_nro, carteira, vencimento, valor, NULL};

	barras_make(barras, 8, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_itau_198(char *carteira, char *nosso_nro, char *nro_doc, char *cod_cedente, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "7", carteira, nosso_nro, nro_doc, cod_cedente, vencimento, valor, NULL};

	barras_make(barras, 8, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_panamericano(char *agencia, char *carteira, char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "8", agencia, carteira, cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 8, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_citi(char *carteira, char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "9", carteira, cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_cef_sigcb(char *carteira, char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "10", carteira, cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_bgn(char *carteira, char *cod_cedente, char *nosso_nro, char *parcela, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "11", carteira, cod_cedente, nosso_nro, parcela, vencimento, valor, NULL};

	barras_make(barras, 8, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_real(char *agencia, char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "12", agencia, cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 7, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_hsbc(char *cod_cedente, char *nosso_nro, char *vencimento, char *valor)
{
	char *barras = if_malloc(BARRAS_LEN);
	char *argv[] = {NULL, "13", cod_cedente, nosso_nro, vencimento, valor, NULL};

	barras_make(barras, 6, argv);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras2linha(char *barras)
{
	char *ld = if_malloc(BARRAS_LEN);

	calc_linha_digitavel(barras, ld);

	return(ld);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * linha2barras(char *linha)
{
	char ld[BARRAS_LEN];
	char *barras = if_malloc(BARRAS_LEN);
	char *p, *q, len;

	if ((linha == NULL) || (strlen(linha) < 54))
	{
		barras[0] = 0;
		return(barras);
	}

	strncpy(ld, linha, BARRAS_LEN);
	memset(barras, 0, BARRAS_LEN);

	core_purge_spaces(ld);

	// banco + moeda '9'
	memcpy(barras, ld, 4);

	// digito
	barras[4] = ld[32];

	// vencimento + valor
	strcpy(barras + 5, ld + 33);

	// bloco 1
	p = barras + 19;
	q = ld + 4;
	len = 5;
	memcpy(p, q, len);

	// bloco 2
	p += len;
	q += len + 1;
	len = 10;
	memcpy(p, q, len);

	// bloco 3
	p += len;
	q += len + 1;
	len = 10;
	memcpy(p, q, len);

	return(barras);
}
// ///////////////////////////////////////////////////////////////////// //


// ///////////////////////////////////////////////////////////////////// //
_PRIVATE char * barras_DAC(char *ln, char (*calc)(char *, int, int))
{
	int len = strlen(ln);
	char buf[2];

	snprintf(buf, 2, "%c", calc(ln, 0, len));
	return(if_strdup(buf));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC10(char *ln)
{
	return(barras_DAC(ln, calc_DAC10));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC11(char *ln)
{
	return(barras_DAC(ln, calc_DAC11));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC11_bradesco(char *ln)
{
	return(barras_DAC(ln, calc_DAC11_bradesco));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC11_santander(char *ln)
{
	return(barras_DAC(ln, calc_DAC11_santander));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC11_citi(char *ln)
{
	return(barras_DAC(ln, calc_DAC11_citi));
}
// ///////////////////////////////////////////////////////////////////// //
_PUBLIC char * barras_DAC11_cef(char *ln)
{
	return(barras_DAC(ln, calc_DAC11_cef));
}
// ///////////////////////////////////////////////////////////////////// //


#ifdef STANDALONE

// ////////////////////////////////////////////////////////////////////////
_PRIVATE int barras_bancos_help(char **argv)
{
	int i, j;

	for (i = 0 ; bancos[i].name != NULL ; i++)
	{
		fprintf(stderr, "\t### %s ###\n", bancos[i].name);
		fprintf(stderr, "\tshell$ %s %d", argv[0], i + 1);
		for (j = 1 ; bancos[i].boleto[j].tag != NULL ; j++)
			fprintf(stderr, " <%s (%ld)>", bancos[i].boleto[j].tag, bancos[i].boleto[j].len);
		fprintf(stderr, "\n\n");
	}

	return(i);
}
// ////////////////////////////////////////////////////////////////////////

// ////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{
	char barras[BARRAS_LEN];
	char ld[BARRAS_LEN];
	int r = 0;

	if (argc < 2)
	{
		fprintf(stderr, "\niFractal - Gerador de Codigo de Barras\n\n");
		fprintf(stderr, "Bancos:\n");
		barras_bancos_help(argv);
		fprintf(stderr, "\n");

		return(1);
	} 

	if (barras_make(barras, argc, argv))
		return(2);

	printf("%s\n", barras);

	calc_linha_digitavel(barras, ld);
	printf("%s\n", ld);

	//if (argc > CMD_LEN)
	//	printf("DAC: %c\n", barras_DAC10(argv[CMD_LEN]));

	return(r);
}
// ////////////////////////////////////////////////////////////////////////

#endif

